% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AgeS_Computation.R
\name{AgeS_Computation}
\alias{AgeS_Computation}
\title{Bayesian analysis for OSL age estimation of various samples}
\usage{
AgeS_Computation(
  DATA,
  SampleNames = DATA$SampleNames,
  Nb_sample = DATA$Nb_sample,
  PriorAge = rep(c(0.01, 100), Nb_sample),
  BinPerSample = rep(1, Nb_sample),
  SavePdf = FALSE,
  OutputFileName = c("MCMCplot", "summary"),
  OutputFilePath = c(""),
  SaveEstimates = FALSE,
  OutputTableName = c("DATA"),
  OutputTablePath = c(""),
  THETA = c(),
  sepTHETA = c(","),
  StratiConstraints = c(),
  sepSC = c(","),
  LIN_fit = TRUE,
  Origin_fit = FALSE,
  distribution = c("cauchy"),
  model = NULL,
  Iter = 10000,
  burnin = 4000,
  adapt = 1000,
  t = 5,
  n.chains = 3,
  jags_method = "rjags",
  autorun = FALSE,
  quiet = FALSE,
  roundingOfValue = 3,
  ...
)
}
\arguments{
\item{DATA}{(\strong{required}) Two inputs are possible:
(1): DATA \link{list} of objects: \code{LT}, \code{sLT}, \code{ITimes}, \code{dLab}, \code{ddot_env}, \code{regDose}, \code{J}, \code{K}, \code{Nb_measurement},
provided by the function \link{Generate_DataFile}, \link{Generate_DataFile_MG} or \link{combine_DataFiles}.
\code{DATA} contains informations for more than one sample.
If there is stratigraphic relations between samples, informations in DATA must be ordered by order of increasing ages.
See the details section to for more informations.
(2): An object of class \code{BayLum.list} which is provided by the output of \link{AgeS_Computation}. When input of class \code{BayLum.list} is identified, no new JAGS model is created. Instead, the JAGS model specified by the \link{AgeS_Computation} output is extended. Useful for when convergence was not originally achieved and a complete restart is not desirable.}

\item{SampleNames}{\link{character} vector: names of samples. The length of this vector is equal to \code{Nb_sample}.}

\item{Nb_sample}{\link{integer}: number of samples, \code{Nb_sample>1}.}

\item{PriorAge}{\link{numeric} vector (with default): lower and upper bounds for age parameter of each sample (in ka).
Note that, \code{length(PriorAge)=2*Nb_sample}
and \code{PriorAge[2i-1,2i]} corresponds to the lower and upper bounds of sample whose number ID is equal to \code{i}.}

\item{BinPerSample}{\link{integer} vector (with default): vector with the number of BIN files per sample.
The length of this vector is equal to \code{Nb_sample}.
\code{BinPerSample[i]} corresponds to the number of BIN files for the sample whose number ID is equal to \code{i}.
For more information to fill this vector, we refer to details in \link{Generate_DataFile} or in \link{Generate_DataFile_MG}.}

\item{SavePdf}{\link{logical} (with default): if TRUE save graphs in pdf file named \code{OutputFileName} in folder \code{OutputFilePath}.}

\item{OutputFileName}{\link{character} (with default): name of the pdf file that will be generated by the function if \code{SavePdf = TRUE};
\code{length(OutputFileName)=2}, see \bold{PLOT OUTPUT} in \bold{Value} section for more informations.}

\item{OutputFilePath}{\link{character} (with default): path to the pdf file that will be generated by the function if \code{SavePdf}=TRUE.
If it is not equal to "", it must be terminated by "/".}

\item{SaveEstimates}{\link{logical} (with default): if TRUE save Bayes' estimates, credible interval at level 68\% and 95\%  and
the result of the Gelman en Rubin test of convergence, in a csv table named \code{OutputFileName} in folder \code{OutputFilePath}.}

\item{OutputTableName}{\link{character} (with default): name of the table that will be generated by the function if \code{SaveEstimates = TRUE}.}

\item{OutputTablePath}{\link{character} (with default): path to the table that will be generated by the function if \code{SaveEstimates = TRUE}.
If it is not equal to "", it must be terminated by "/".}

\item{THETA}{\link{numeric} \link{matrix} or \link{character} (with default): input object for systematic and individual error.
If systematic errors are considered, see the details section for instructions regarding how to correctly fill \code{THETA};
the user can refer to a matrix (numeric matrix) or to a csv file (character).
Otherwise, default value is suitable, and only individual errors are considered.}

\item{sepTHETA}{\link{character} (with default): if \code{THETA} is character,
indicate column separator in \code{THETA} CSV-file.}

\item{StratiConstraints}{numeric matrix or character(with default): input object for the stratigraphic relation between samples. If there is stratigraphic relation between samples see the details section for instructions regarding how to correctly fill \code{StratiConstraints}; the user can refer to a matrix (numeric matrix) or to a csv file (character).
If there is no stratigraphic relation default value is suitable.}

\item{sepSC}{\link{character} (with default): if \code{StratiConstraints} is character,
indicate column separator in \code{StratiConstraints} .csv file.}

\item{LIN_fit}{\link{logical} (with default): if TRUE (default) allows a linear component,
on top of the (default) saturating exponential curve, for the fitting of dose response curves.
See details section for more informations on the proposed dose response curves.}

\item{Origin_fit}{\link{logical} (with default): if TRUE, forces the dose response curves to pass through the origin.
See details section for more informations on the proposed growth curves.}

\item{distribution}{\link{character} (with default): type of distribution that defines
how individual equivalent dose values are distributed around the palaeodose.
Allowed inputs are \code{"cauchy"}, \code{"gaussian"}, \code{"lognormal_A"} and \code{"lognormal_M"},
see details section for more informations.}

\item{model}{\link{character} (\emph{optional}): allows to provide a custom model to the function as text string. Please note that if this option is chosen the parameter \code{distribution} is ignored and no safety net is applied. If the function crashes it is up to the user.}

\item{Iter}{\link{integer} (with default): the number of iterations to run which will be used to assess convergence and ages (see \link[runjags:run.jags]{runjags::run.jags}).}

\item{burnin}{\link{integer} (with default): the number of iterations used to "home in" on the stationary posterior distribution. These are not used for assessing convergence (see \link[runjags:run.jags]{runjags::run.jags}).}

\item{adapt}{\link{integer} (with default): the number of iterations used in the adaptive phase of the simulation (see \link[runjags:run.jags]{runjags::run.jags}).}

\item{t}{\link{integer} (with default): 1 every \code{t} iterations of the MCMC is considered for sampling the posterior distribution.
(for more information see \link[runjags:run.jags]{runjags::run.jags}).}

\item{n.chains}{\link{integer} (with default): number of independent chains for the model (for more information see \link[runjags:run.jags]{runjags::run.jags}).}

\item{jags_method}{\link{character} (with default): select which method to use in order to call JAGS. jags_methods \code{"rjags"} (the default) and \code{"rjparallel"} have been tested. (for more information about these possibilities and others, see \link[runjags:run.jags]{runjags::run.jags})}

\item{autorun}{\link{logical} (with default): choose to automate JAGS processing. JAGS model will be automatically extended until convergence is reached (for more information see \link[runjags:autorun.jags]{runjags::autorun.jags}).}

\item{quiet}{\link{logical} (with default): enables/disables \code{rjags} messages}

\item{roundingOfValue}{\link{integer} (with default):  Integer indicating the number of decimal places to be used, default = 3.}

\item{...}{further arguments that can be passed to control the Bayesian process. 1) When creating a new JAGS model, see details
for supported arguments. 2) If extending a JAGS model see arguments of \link[runjags:run.jags]{runjags::extend.JAGS}.}
}
\value{
\strong{NUMERICAL OUTPUT}

\enumerate{
\item \strong{A list of type \code{BayLum.list} containing the following objects:}
\itemize{
\item \bold{Sampling}: that corresponds to a sample of the posterior distributions
of the age (in ka), palaeodose (in Gy) and equivalent dose dispersion (in Gy) parameters for each sample;
\item \bold{Model_GrowthCurve}: stating which dose response fitting option was chosen;
\item \bold{Distribution}: stating which distribution was chosen to model the dispersion of
individual equivalent dose values around the palaeodose of the sample;
\item \bold{PriorAge}: stating the priors used for the age parameter (in ka);
\item \bold{StratiConstraints}: stating the stratigraphic relations between samples considered in the model;
\item \bold{CovarianceMatrix}: stating the covariance matrix of error used in the model, highlighting common errors between samples or not.
\item \bold{model}: returns the model that was used for the Bayesian modelling as a \link{character}
\item \bold{JAGS model output}: returns the JAGS model with class "runjags".
}
\item\bold{The Gelman and Rubin test of convergency}: prints the result of the Gelman and Rubin test of convergence for
the age, palaeodose and equivalent dose dispersion parameters for each sample.
A result close to one is expected.\cr
In addition, the user must visually assess the convergence of the trajectories by looking at the graph
generated by the function (see \bold{PLOT OUTPUT} for more informations).\cr
If both convergences (Gelman and Rubin test and plot checking) are satisfactory,
the user can consider the estimates as valid.
Otherwise, the user may try increasing the number of MCMC iterations (\code{Iter})
or being more precise on the \code{PriorAge} parameter (for example specify if it is a young sample \code{c(0.01,10)} an old sample \code{c(10,100)}),
or changing the parameter \code{distribution} or the growth curve, to reach convergence.
\item \bold{Credible intervals and Bayes estimates}: prints the Bayes estimates, the credible intervals at 95\% and 68\% for
the age, palaeodose and equivalent dose dispersion parameters for each sample.
}

\strong{PLOT OUTPUT}

\enumerate{
\item\bold{MCMC trajectories}: A graph with the MCMC trajectories and posterior distributions of the age, palaeodose and equivalent dose dispersion parameters
is displayed, there is one page per sample. \cr
The first line of the figure corresponds to the age parameter, the second to the palaeodose parameter
and the third to the equivalent dose dispersion parameter.
On each line, the plot on the left represents the MCMC trajectories, and the one on the right the posterior distribution of the parameter.
\item \bold{Summary of sample age estimates}: plot credible intervals and Bayes estimate of each sample age on a same graph.
}

To give the results in a publication, we recommend to give the Bayes' estimate of the parameters as well as the credible interval at 95\% or 68\%.
}
\description{
This function computes the age (in ka) of at least two samples
according to the model developed in Combès and Philippe (2017),
based on outputs of \link{Generate_DataFile} or \link{Generate_DataFile_MG}
or both of them using \link{combine_DataFiles}.\cr
Samples, for which data is available in several BIN files, can be analysed.\cr
Single-grain or Multi-grain OSL measurements can be analysed simultaneously.
}
\details{
\strong{Supported \code{...} arguments}
\tabular{lllll}{
ARGUMENT \tab INPUT \tab METHOD \tab DEFAULT \tab DESCRIPTION\cr
\code{max.time} \tab \link{character} \tab \code{rjparallel} \tab \code{Inf} \tab maximum allowed processing time, e.g.,
\verb{10m} for 10 minutes (cf. \link[runjags:autorun.jags]{runjags::autorun.jags})\cr
\code{interactive} \tab \link{logical} \tab \code{rjparallel} \tab \code{FALSE} \tab enable/disable interactive mode (cf. \link[runjags:autorun.jags]{runjags::autorun.jags})\cr
\code{startburnin} \tab \link{integer} \tab \code{rjparallel} \tab  \code{4000} \tab number of burn-in iterations (cf. \link[runjags:autorun.jags]{runjags::autorun.jags}) \cr
\code{startsample} \tab \link{integer} \tab \code{rjparallel} \tab \code{10000} \tab total number of samples to assess convergence
(cf. \link[runjags:autorun.jags]{runjags::autorun.jags}) \cr
\code{inits} \tab named \link{list} \tab \code{rjparallel} \tab \code{NA} \tab fine control over random seeds and random number generator \link[runjags:autorun.jags]{runjags::autorun.jags}
}

\strong{How to fill \code{StratiConstraints}}\cr

If there is stratigraphic relations between samples,
\emph{informations in DATA must be ordered by order of increasing ages}.
To do this the user can either fill right \code{Names} in \link{Generate_DataFile}
or in \link{Generate_DataFile_MG} (as it is indicated in Details section of these function),
or ordered by order of increasing ages outputs of \link{Generate_DataFile}
or \link{Generate_DataFile_MG} in \link{combine_DataFiles}

The user can fill the \code{StratiConstraints} matrix as follow.
\enumerate{
\item \bold{Size of the matrix}: row number of \code{StratiConstraints} matrix is equal to \code{Nb_sample+1},
and column number is equal to \code{Nb_sample}.
\item \bold{First line of the matrix}:
for all \code{i in {1,...,Nb_Sample}}, \code{StratiConstraints[1,i]=1} that means the lower bound of the sample age (given in \code{PriorAge[2i-1]})
for the sample whose number ID is equal to \code{i}, is taken into account.
\item \bold{Sample relations}: for all  \code{j in {2,...,Nb_Sample+1}} and all \code{i in {j,...,Nb_Sample}},
\code{StratiConstraints[j,i]=1} if sample age whose number ID is equal to \code{j-1} is lower than sample age whose number ID is equal to \code{i}.
Otherwise, \code{StratiConstraints[j,i]=0}.
}
Note that \code{StratiConstraints_{2:Nb_sample+A,1:Nb_sample}} is a upper triangular matrix.

The user can also use \code{\link{SCMatrix}} or \code{\link{SC_Ordered}} (if all samples are ordered) functions
to construct the \code{StratiConstraints} matrix.

The user can also refer to a csv file that contains the relation between samples as defined above.
The user must take care about the separator used in the csv file using the argument \code{sepSC}.\cr

\strong{How to fill \code{THETA} covariance matrix concerning common and individual error?}\cr

If systematic errors are considered, the user can fill the \code{THETA} matrix as follows.
\itemize{
\item row number of \code{THETA} is equal the column number, equal to \code{Nb_sample}.
\item For all \code{i in {1,...,Nb_sample}}, \code{THETA[i,i]} contains individual error
plus systematic error of the sample whose number ID is equal to \code{i}.
\item For all \code{i,j in {1,...,Nb_sample}} and \code{i} different from \code{j} ,
\code{THETA[i,j]} contains common error between samples whose number ID are equal to \code{i} and \code{j}.
}
Note that \code{THETA[i,j]} is a symetric matrix.

The user can also refer to a .csv file that contains the errors as defined above.\cr

Alternatively you can use the function \link{create_ThetaMatrix}.

\strong{Option on growth curves}\cr

As for \code{\link{Age_Computation}} and \code{\link{Palaeodose_Computation}}, the user can choose from 4 dose response curves:
\itemize{
\item \bold{Saturating exponential plus linear growth} (\code{AgesMultiCS2_EXPLIN}):

for all \code{x} in IR+, \code{f(x)=a(1-exp(-x/b))+cx+d}; select
\itemize{
\item \code{LIN_fit=TRUE}
\item \code{Origin_fit=FALSE}
}
\item \bold{Saturating exponential growth} (\code{AgesMultiCS2_EXP}):

for all \code{x} in IR+, \code{f(x)=a(1-exp(-x/b))+d}; select
\itemize{
\item \code{LIN_fit=FALSE}
\item \code{Origin_fit=FALSE}
}
\item \bold{Saturating exponential plus linear growth and fitting
through the origin} (\code{AgesMultiCS2_EXPLINZO}):

for all \code{x} in IR+, \code{f(x)=a(1-exp(-x/b))+cx}; select
\itemize{
\item \code{LIN_fit=TRUE}
\item \code{Origin_fit=TRUE}
}
\item \bold{Saturating exponential growth and fitting through the origin} (\code{AgesMultiCS2_EXPZO}):

for all \code{x} in IR+, \code{f(x)=a(1-exp(-x/b))}; select
\itemize{
\item \code{LIN_fit=FALSE}
\item \code{Origin_fit=TRUE}
}
}

\strong{Option on equivalent dose distribution around the palaeodose}\cr

The use can choose between :
\itemize{
\item \code{cauchy}: a Cauchy distribution with location parameter equal to the palaeodose of the sample;
\item \code{gaussian}: a Gaussian distribution with mean equal to the palaeodose of the sample;
\item \code{lognormal_A}: a log-normal distribution with mean or \bold{A}verage equal to the palaeodose of the sample:
\item \code{lognormal_M}: a log-normal distribution with \bold{M}edian equal to the palaeodose of the sample.
}
}
\note{
Please note that the initial values for all MCMC are currently all the same for all chains since we rely on the automatic
initial value generation of JAGS. This is not optimal and will be changed in future. However, it does not affect the quality
of the age estimates if the chains have converged.
}
\examples{
## Age computation of samples GDB5 and GDB3,
## load data
data(DATA2) # GD85
data(DATA1) # GD83

## produce DataFile
Data <- combine_DataFiles(DATA2, DATA1)

## without common error, assuming GDB5 age younger than GDB3 age
Nb_sample <- 2
SC <- matrix(
  data = c(1,1,0,1,0,0),
  ncol = 2,
  nrow = (Nb_sample+1),
  byrow = TRUE)

\dontrun{
## run standard
Age <- AgeS_Computation(
  DATA = Data,
  Nb_sample = Nb_sample,
  SampleNames = c("GDB5","GDB3"),
  PriorAge = rep(c(1,100), 2),
  StratiConstraints = SC,
  Iter = 100,
  quiet = FALSE,
  jags_method = "rjags"
)

## extend model
Age_extended <- AgeS_Computation(
  DATA = Age,
  Nb_sample = Nb_sample,
  SampleNames = c("GDB5","GDB3"),
  PriorAge = rep(c(1,100), 2),
  StratiConstraints = SC,
  adapt = 0,
  burnin = 500,
  Iter = 1000,
  quiet = FALSE,
  jags_method = "rjags"
)

## autorun mode
Age <- AgeS_Computation(
  DATA = Data,
  Nb_sample = Nb_sample,
  SampleNames = c("GDB5","GDB3"),
  PriorAge = rep(c(1,100), 2),
  StratiConstraints = SC,
  Iter = 10000,
  quiet = FALSE,
  jags_method = "rjags",
  autorun = TRUE)

## parallel mode
Age <- AgeS_Computation(
  DATA = Data,
  Nb_sample = Nb_sample,
  SampleNames = c("GDB5","GDB3"),
  PriorAge = rep(c(1,100), 2),
  StratiConstraints = SC,
  Iter = 10000,
  quiet = FALSE,
  jags_method = "rjparallel")
}

}
\references{
Combes, Benoit and Philippe, Anne, 2017.
Bayesian analysis of multiplicative Gaussian error for multiple ages estimation in optically stimulated luminescence dating.
Quaternary Geochronology (39, 24-34)

Combes, B., Philippe, A., Lanos, P., Mercier, N., Tribolo, C., Guerin, G., Guibert, P., Lahaye, C., 2015.
A Bayesian central equivalent dose model for optically stimulated luminescence dating.
Quaternary Geochronology 28, 62-70. doi:10.1016/j.quageo.2015.04.001
}
\seealso{
\link{Generate_DataFile}, \link{Generate_DataFile_MG}, \link[rjags:rjags-package]{rjags::rjags-package}, \link{plot_MCMC}, \link{SCMatrix}, \link{Age_Computation}, \link{Palaeodose_Computation}, \link{plot_Ages}, \link{create_ThetaMatrix}, \link[runjags:autorun.jags]{runjags::autorun.jags}
}
\author{
Claire Christophe, Anne Philippe, Guillaume Guérin, Sebastian Kreutzer
}
