generateHMM <- function(n, J, obsdist, obspar, Pi, delta = NULL, seed = NULL) {
  # Set seed if provided (but don't overwrite if already set at higher level)
  if (!is.null(seed)) {
    set.seed(seed)
  }

  # Calculate stationary distribution if initial probabilities not provided
  if (is.null(delta)) {
    delta <- solve(t(diag(J) - Pi + 1), rep(1, J))
  }

  # Initialize state vector and sample first state
  mvect <- 1:J
  state <- numeric(n)
  state[1] <- sample(mvect, 1, prob = delta)

  # Generate state sequence using transition probabilities
  for (i in 2:n) {
    state[i] <- sample(mvect, 1, prob = Pi[state[i - 1], ])
  }

  # Generate observations based on specified distribution
  if (obsdist == "norm") {
    # Normal distribution
    if (is.null(obspar$mean)||is.null(obspar$sd)) {
      stop("Missing parameters 'mean' and/or 'sd' for normal distribution.")
    }
    x <- rnorm(n, mean = obspar$mean[state], sd = obspar$sd[state])
  } else if (obsdist == "pois") {
    # Poisson distribution
    if (is.null(obspar$lambda)) {
      stop("Missing parameter 'lambda' for Poisson distribution.")
    }
    x <- rpois(n, lambda = obspar$lambda[state])
  } else if (obsdist == "weibull") {
    # Weibull distribution
    if (is.null(obspar$shape)||is.null(obspar$scale)) {
      stop("Missing parameters 'shape' and/or 'scale' for Weibull distribution.")
    }
    x <- rweibull(n, shape = obspar$shape[state], scale = obspar$scale[state])
  } else if (obsdist == "zip") {
    # Zero-inflated Poisson distribution
    if (is.null(obspar$pi)||is.null(obspar$lambda)) {
      stop("Missing parameters 'pi' and/or 'lambda' for ZIP distribution.")
    }
    x <- numeric(n)
    for (i in 1:n) {
      if (runif(1) < obspar$pi[state[i]]) {
        x[i] <- 0 # Zero with probability pi
      } else {
        x[i] <- rpois(1, lambda = obspar$lambda[state[i]]) # Poisson otherwise
      }
    }
  } else if (obsdist == "nbinom") {
    # Negative binomial distribution
    if (is.null(obspar$mu)||is.null(obspar$size)) {
      stop("Missing parameters 'size' and/or 'mu' for negative binomial distribution.")
    }
    x <- rnbinom(n, size = obspar$size[state], mu = obspar$mu[state])
  } else if(obsdist=="zinb"){
    # Zero-inflated negative binomial distribution
    if(is.null(obspar$size)||is.null(obspar$mu)||is.null(obspar$pi)){
      stop("Missing parameters 'size' and/or 'mu' and/or 'pi' for the zero-inflated negative binomial distribution")
    }
    x <- numeric(n)
    for (i in 1:n) {
      if (runif(1) < obspar$pi[state[i]]) {
        x[i] <- 0 # Zero with probability pi
      } else {
        x[i] <- rnbinom(1, size = obspar$size[state[i]], mu=obspar$mu[state[i]])
      }
    }
  } else if (obsdist == "exp"){
    # Exponential distribution
    if(is.null(obspar$rate)){
      stop("Missing parameter 'rate' for the exponential distribution")
    }
    x<- rexp(n, rate = obspar$rate[state])
  } else if (obsdist == "gamma"){
    # Gamma distribution
    if(is.null(obspar$shape)||is.null(obspar$rate)){
      stop("Missing parameters 'shape' and/or 'rate' for the gamma distribution")
    }
    x <- rgamma(n, shape=obspar$shape[state], rate=obspar$rate[state])
  } else if(obsdist == "lnorm"){
    # Log-normal distribution
    if(is.null(obspar$meanlog)||is.null(obspar$sdlog)){
      stop("Missing parameters 'meanlog' and/or 'sdlog' for the log-normal distribution")
    }
    x <- rlnorm(n, meanlog=obspar$meanlog[state], sdlog=obspar$sdlog[state])
  } else if(obsdist=="gev"){
    # Generalized extreme value distribution
    if(is.null(obspar$loc)||is.null(obspar$scale)||is.null(obspar$shape)){
      stop("Missing parameters 'loc' and/or 'scale' and/or 'shape' for the generalised extreme value distribution")
    }
    x <- revd(n, loc=obspar$loc[state], scale=obspar$scale[state], shape=obspar$shape[state], type="GEV")
  } else if(obsdist=="ZInormal"){
    # Zero-inflated normal distribution
    if(is.null(obspar$mean)||is.null(obspar$sd)||is.null(obspar$pi)){
      stop("Missing parameters 'mean' and/or 'sd' and/or pi for the zero-inflated normal distribution")
    }
    x <- numeric(n)
    for (i in 1:n) {
      if (runif(1) < obspar$pi[state[i]]) {
        x[i] <- 0 # Zero with probability pi
      } else {
        x[i] <- rnorm(1, mean = obspar$mean[state[i]], sd=obspar$sd[state[i]])
      }
    }
  } else if(obsdist=="ZIgamma"){
    # Zero-inflated gamma distribution
    if(is.null(obspar$shape)||is.null(obspar$rate)||is.null(obspar$pi)){
      stop("Missing parameters 'shape' and/or 'rate' and/or pi for the zero-inflated gamma distribution")
    }
    x <- numeric(n)
    for (i in 1:n) {
      if (runif(1) < obspar$pi[state[i]]) {
        x[i] <- 0 # Zero with probability pi
      } else {
        x[i] <- rgamma(1, shape = obspar$shape[state[i]], rate=obspar$rate[state[i]])
      }
    }
  } else {
    stop("Distribution not supported")
  }

  return(list(x=x, state=state))
}

