\name{HSMMVarianceMatrix}
\alias{HSMMVarianceMatrix}
\title{Variance-Covariance Matrix for Hidden Semi-Markov Models}
\description{
Computes an approximate variance-covariance matrix of the parameter estimates
from a fitted Hidden Semi-Markov Model (HSMM) using a numerical Hessian of the log-likelihood.
The Hessian is inverted and adjusted to ensure positive-definiteness, providing
an estimate of parameter uncertainty.
}
\usage{
HSMMVarianceMatrix(x, HSMM, M = NA, obsdist, dwelldist,
                   h = 1e-5, method = "central", verbose = TRUE)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}
\item{HSMM}{A fitted HSMM object (typically the output from \code{\link{findmleHSMM}}),
containing estimated model parameters \code{Pi}, \code{delta},
observation parameters, and dwell-time parameters.}
\item{M}{Integer. Maximum dwell time to consider for semi-Markov states.
If \code{NA}, defaults to \code{min(length(x), 1000)}.}
\item{obsdist}{Character string. Observation distribution. Supported distributions:
"norm", "pois", "weibull", "nbinom", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}
\item{dwelldist}{Character string. Dwell-time distribution. Supported distributions:
"pois", "nbinom", "geom".}
\item{h}{Numeric. Step size for finite-difference approximation of the Hessian.
Default is \code{1e-5}.}
\item{method}{Character string. Method for numerical Hessian computation.
Options are \code{"central"} (default, central differences) or \code{"forward"} (forward differences).}
\item{verbose}{Logical, if \code{TRUE} (default), progress messages are printed to the console.
Set to \code{FALSE} to suppress informational output.}
}
\value{
A variance-covariance matrix of the HSMM parameter estimates, computed as the
inverse of the observed information matrix (negative Hessian of the log-likelihood),
adjusted to be positive-definite.
}
\details{
This function approximates the observed Fisher information for an HSMM by
numerically differentiating the log-likelihood with respect to the model parameters.
The resulting Hessian is inverted to obtain a variance-covariance matrix.
Since numerical Hessians can yield non-positive-definite matrices due to
rounding error or flat likelihood surfaces, the eigenvalues are adjusted
to enforce positive-definiteness.
}
\examples{
# Example with 2-state Normal HSMM
J <- 2
Pi <- matrix(c(0, 1,
               1, 0), nrow = 2, byrow = TRUE)
obspar <- list(mean = c(0, 3), sd = c(1, 1))
dwellpar <- list(lambda = c(5, 7))
# Simulate data
sim_data <- generateHSMM(n = 100, J = J, obsdist = "norm",
                         dwelldist = "pois", obspar = obspar,
                         dwellpar = dwellpar, Pi = Pi)
# Fit HSMM
HSMM_fit <- findmleHSMM(x = sim_data$x, J = J, M = 50,
                        obsdist = "norm", dwelldist = "pois",
                        obspar = obspar, dwellpar = dwellpar, Pi = Pi)
# Compute variance-covariance matrix
vcov_matrix <- HSMMVarianceMatrix(x = sim_data$x, HSMM = HSMM_fit,
                                  obsdist = "norm", dwelldist = "pois",
                                  M = 50)
vcov_matrix
}
\seealso{
\code{\link{generateHSMM}}, for simulating HSMM data.
\code{\link{findmleHSMM}}, for fitting Hidden Semi-Markov Models.
\code{\link{HMMVarianceMatrix}}, for numerically computing the variance-covariance matrix of a fitted hidden Markov model.
}
\author{Aimee Cody}

