\name{findmleHSMM}
\alias{findmleHSMM}
\title{Maximum Likelihood Estimation for Hidden Semi-Markov Models}
\description{
Estimates the parameters of a Hidden Semi-Markov Model (HSMM) using the EM algorithm with specified observation and dwell-time distributions.
}
\usage{
findmleHSMM(x, J, M = NA, obsdist, dwelldist, obspar, dwellpar,
            Pi, delta = NULL, maxiter = 100, tol = 1e-05, shift = FALSE,
            verbose = TRUE, seed = NULL)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}

\item{J}{Integer. The number of hidden states in the model. Must be greater than 1.}

\item{M}{Integer. Maximum dwell time to consider for semi-Markov states. Defaults to \code{min(length(x), 1000)} if \code{NA}.}

\item{obsdist}{Character string. Observation distribution. Supported distributions: "norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}

\item{dwelldist}{Character string. Dwell-time distribution. Supported distributions: "pois", "nbinom", "betabinom", "geom".}

\item{obspar}{List. Parameters for the observation distribution. Each parameter must be a vector of length \code{J}, with values for each state. Required parameters:
  \itemize{
    \item \code{norm}: \code{mean}, \code{sd}
    \item \code{pois}: \code{lambda}
    \item \code{weibull}: \code{shape}, \code{scale}
    \item \code{zip}: \code{lambda}, \code{pi}
    \item \code{nbinom}: \code{size}, \code{mu}
    \item \code{zinb}: \code{size}, \code{mu}, \code{pi}
    \item \code{exp}: \code{rate}
    \item \code{gamma}: \code{shape}, \code{rate}
    \item \code{lnorm}: \code{meanlog}, \code{sdlog}
    \item \code{gev}: \code{loc}, \code{scale}, \code{shape}
    \item \code{ZInormal}: \code{mean}, \code{sd}, \code{pi}
    \item \code{ZIgamma}: \code{shape}, \code{rate}, \code{pi}
  }
}

\item{dwellpar}{List. Parameters for the dwell-time distribution. Each parameter must be a vector of length \code{J}, with values for each state. Required parameters:
  \itemize{
    \item \code{pois}: \code{lambda}
    \item \code{nbinom}: \code{size}, \code{mu}
    \item \code{betabinom}: \code{size}, \code{prob}
    \item \code{geom}: \code{prob}
  }
}

\item{Pi}{Matrix. The \code{J x J} transition probability matrix between states. Rows must sum to 1.}

\item{delta}{Numeric vector of length \code{J}. Initial state distribution. If \code{NULL}, the stationary distribution of \code{Pi} is used.}

\item{maxiter}{Integer. Maximum number of EM iterations. Defaults to 100.}

\item{tol}{Numeric. Convergence tolerance for the change in log-likelihood. Defaults to 1e-05.}

\item{shift}{Logical. Indicates whether the dwell-time distribution includes a user-specified shift parameter. If \code{TRUE}, the distribution uses the provided shift parameter. If \code{FALSE}, the dwell-time distribution is automatically shifted by 1. Defaults to \code{FALSE}.}

\item{verbose}{Logical. If \code{TRUE}, progress messages including iteration numbers and log-likelihood values are printed to the console. Default is \code{TRUE}.}

\item{seed}{Integer or \code{NULL}. Random seed for reproducibility. Default is \code{NULL}.}
}
\value{
A list containing:
\item{loglikelihoods}{Numeric vector of log-likelihood values across iterations.}
\item{AIC}{Akaike Information Criteria for the fitted model.}
\item{BIC}{Bayesian Information Criteria for the fitted model.}
\item{delta}{Numeric vector. Estimated initial state probabilities.}
\item{Pi}{Matrix. Estimated \code{J x J} transition probability matrix.}
\item{dwellparameters}{List. Estimated dwell-time distribution parameters.}
\item{observationparameters}{List. Estimated observation distribution parameters.}
}
\details{
This function fits a Hidden Semi-Markov Model to a sequence of observations using an EM algorithm. At each iteration, the algorithm:
\itemize{
  \item Computes observation probabilities for each state
  \item Calculates dwell-time probabilities and survival functions
  \item Performs a forward-backward-like procedure to estimate expected sufficient statistics
  \item Re-estimates observation and dwell-time parameters
}

Supported observation distributions include normal, Poisson, Weibull, zero-inflated Poisson (ZIP), negative binomial, zero-inflated negative binomial (ZINB), exponential, gamma, log-normal, generalized extreme value (GEV), zero-inflated normal, and zero-inflated gamma.

Supported dwell-time distributions include Poisson, negative binomial, beta-binomial, and geometric.

When \code{verbose = TRUE}, the function displays log-likelihood values at each iteration and reports whether convergence was achieved.
}
\examples{

J <- 3
Pi <- matrix(c(0.0, 0.6, 0.4,
               0.5, 0.0, 0.5,
               0.3, 0.7, 0.0), nrow = J, byrow = TRUE)

obspar <- list(
  mean = c(-2, 0, 3),
  sd   = c(1, 1.5, 2)
)

dwellpar <- list(
  lambda = c(3, 5, 4)
)

sim <- generateHSMM(n = 1000, J = J, obsdist = "norm",
                    dwelldist = "pois", obspar = obspar,
                    dwellpar = dwellpar, Pi = Pi)

fit <- findmleHSMM(x = sim$x, J = J, M = 100,
                   obsdist = "norm", dwelldist = "pois",
                   obspar = obspar, dwellpar = dwellpar,
                   Pi = Pi)

fit$observationparameters
fit$dwellparameters
fit$Pi
fit$delta
fit$loglikelihoods

fit_silent <- findmleHSMM(x = sim$x, J = J, M = 100,
                          obsdist = "norm", dwelldist = "pois",
                          obspar = obspar, dwellpar = dwellpar,
                          Pi = Pi, verbose = FALSE)
}

\seealso{
\code{\link{findmleHMM}} for Hidden Markov Model estimation.
\code{\link{generateHSMM}} for simulating HSMM data.
}
\author{
Aimee Cody
}




