% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_lambda_trait.R
\name{simulate_lambda_trait}
\alias{simulate_lambda_trait}
\title{Simulate Trait Data with Target Phylogenetic Signal (Lambda)}
\usage{
simulate_lambda_trait(
  target_lambda,
  tree,
  max_attempts = 1e+05,
  tolerance = 0.02
)
}
\arguments{
\item{target_lambda}{Numeric. The desired phylogenetic signal strength (lambda value).
Should be between 0 and 1.
- 0: No phylogenetic signal (star phylogeny)
- 1: Full phylogenetic signal (Brownian motion)}

\item{tree}{An object of class "phylo". The phylogenetic tree for trait simulation.}

\item{max_attempts}{Integer. Maximum number of simulation attempts before giving up.
Default is 100000.}

\item{tolerance}{Numeric. Acceptable difference between target and estimated lambda.
Default is 0.02.}
}
\value{
A data.frame with one column named 'trait' containing the simulated trait values.
  Row names correspond to tip labels from the phylogenetic tree. Returns NULL if
  the target lambda cannot be achieved within the specified tolerance and attempts.
}
\description{
This function generates trait data that matches a specified phylogenetic signal
strength (Pagel's lambda) through iterative simulation and testing.
}
\details{
The function works by:

1. Transforming the phylogenetic tree according to the target lambda value using rescale()

2. Simulating trait data using fastBM() on the transformed tree

3. Estimating the phylogenetic signal using phylosig()

4. Repeating until the estimated lambda is within tolerance of the target

Special cases:
- When target_lambda = 0: Sets internal branch lengths to 0, keeping only terminal branches
- When target_lambda = 1: Uses the original tree without transformation
}
\note{
The function may take considerable time to converge for certain lambda values,
especially those close to intermediate values.

Consider adjusting
the tolerance parameter if convergence is slow.

If `target_lambda` is greater than 1,
it will be automatically capped at 1, as lambda values typically range from 0 to 1.
}
\examples{
# Generate a random tree
tree <- ape::rtree(50)

# Simulate trait with strong phylogenetic signal
trait_data <- simulate_lambda_trait(0.8, tree)

# Verify the phylogenetic signal
trait_vector <- setNames(trait_data$trait, rownames(trait_data))
phytools::phylosig(tree, trait_vector, method = "lambda", test = TRUE)

}
