
#' 
#' Restriction Matrix
#' 
#' @description
#' This function returns the restriction matrix (de la Torre, 2011; Ma & de la Torre, 2020) based on two q-vectors, 
#' where the two q-vectors can only differ by one attribute.
#' 
#' @param q1 A q-vector
#' @param q2 Another q-vector
#' @param pattern.reduced.q1 reduced attribute patterns for q1 (Default = \code{NULL})
#' @param pattern.reduced.q2 reduced attribute patterns for q2 (Default = \code{NULL})
#' 
#' @seealso \code{\link[Qval]{Wald.test}}
#' 
#' @return A restriction matrix
#' 
#' @references
#' de la Torre, J. (2011). The Generalized DINA Model Framework. Psychometrika, 76(2), 179-199. DOI: 10.1007/s11336-011-9207-7.
#' 
#' Ma, W., & de la Torre, J. (2020). An empirical Q-matrix validation method for the sequential generalized DINA model. British Journal of Mathematical and Statistical Psychology, 73(1), 142-163. DOI: 10.1111/bmsp.12156.
#' 
#' @examples
#' library(Qval)
#' q1 <- c(1, 1, 0)
#' q2 <- c(1, 1, 1)
#' 
#' Rmatrix <- get.Rmatrix(q1, q2)
#' 
#' print(Rmatrix)
#' 
#' 
#' @export
#' @importFrom GDINA attributepattern
#' @importFrom utils combn
#' 
get.Rmatrix <- function(q1, q2, pattern.reduced.q1 = NULL, pattern.reduced.q2 = NULL) {
  att.posi.i.k <- which(q2 > 0)
  att.posi.i <- which(q1 > 0)
  
  if (is.null(pattern.reduced.q1))
    pattern.reduced.q1 <- attributepattern(length(att.posi.i))
  if (is.null(pattern.reduced.q2))
    pattern.reduced.q2 <- attributepattern(length(att.posi.i.k))
  
  Rmatrix <- matrix(0, nrow(pattern.reduced.q1), nrow(pattern.reduced.q2))
  
  dif.reduced <- which(q1[att.posi.i.k] != q2[att.posi.i.k])
  
  if (length(dif.reduced) > 0)
    pattern.compare <- pattern.reduced.q2[, -dif.reduced, drop = FALSE]
  else
    pattern.compare <- pattern.reduced.q2
  
  keys <- apply(pattern.compare, 1, paste, collapse = "")
  groups <- split(seq_along(keys), keys)
  
  t <- 1L
  for (g in groups) {
    if (length(g) >= 2) {
      comb <- combn(g, 2)
      num_pairs <- ncol(comb)
      idx <- t:(t + num_pairs - 1)
      Rmatrix[idx, comb[1, ]] <- 1
      Rmatrix[idx, comb[2, ]] <- -1
      t <- t + num_pairs
    }
  }
  
  if (t <= nrow(Rmatrix)) Rmatrix <- Rmatrix[1:(t - 1), , drop = FALSE]
  
  return(Rmatrix)
}