% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dist_swdist.R
\name{swdist}
\alias{swdist}
\title{Sliced Wasserstein Distance}
\usage{
swdist(X, Y, p = 2, ...)
}
\arguments{
\item{X}{an \eqn{(M\times P)} matrix of row observations.}

\item{Y}{an \eqn{(N\times P)} matrix of row observations.}

\item{p}{an exponent for the order of the distance (default: 2).}

\item{...}{extra parameters including \describe{
\item{num_proj}{the number of Monte Carlo samples for SW computation (default: 496).}
}}
}
\value{
a named list containing\describe{
\item{distance}{\eqn{\mathcal{SW}_p} distance value.}
\item{projdist}{a length-\code{num_proj} vector of projected univariate distances.}
}
}
\description{
Sliced Wasserstein (SW) Distance is a popular alternative to the standard Wasserstein distance due to its computational 
efficiency on top of nice theoretical properties. For the \eqn{d}-dimensional probability 
measures \eqn{\mu} and \eqn{\nu}, the SW distance is defined as 
\deqn{\mathcal{SW}_p (\mu, \nu) = 
\left( \int_{\mathbb{S}^{d-1}} \mathcal{W}_p^p (
\langle \theta, \mu\rangle, \langle \theta, \nu \rangle) d\lambda (\theta) \right)^{1/p},}
where \eqn{\mathbb{S}^{d-1}} is the \eqn{(d-1)}-dimensional unit hypersphere and 
\eqn{\lambda} is the uniform distribution on \eqn{\mathbb{S}^{d-1}}. Practically, 
it is computed via Monte Carlo integration.
}
\examples{
\donttest{
#-------------------------------------------------------------------
#  Sliced-Wasserstein Distance between Two Bivariate Normal
#
# * class 1 : samples from Gaussian with mean=(-1, -1)
# * class 2 : samples from Gaussian with mean=(+1, +1)
#-------------------------------------------------------------------
# SMALL EXAMPLE
set.seed(100)
m = 20
n = 30
X = matrix(rnorm(m*2, mean=-1),ncol=2) # m obs. for X
Y = matrix(rnorm(n*2, mean=+1),ncol=2) # n obs. for Y

# COMPUTE THE SLICED-WASSERSTEIN DISTANCE
outsw <- swdist(X, Y, num_proj=100)

# VISUALIZE
# prepare ingredients for plotting
plot_x = 1:1000
plot_y = base::cumsum(outsw$projdist)/plot_x

# draw
opar <- par(no.readonly=TRUE)
plot(plot_x, plot_y, type="b", cex=0.1, lwd=2,
     xlab="number of MC samples", ylab="distance",
     main="Effect of MC Sample Size")
abline(h=outsw$distance, col="red", lwd=2)
legend("bottomright", legend="SW Distance", 
       col="red", lwd=2)
par(opar)
}

}
\references{
\insertRef{rabin_2012_WassersteinBarycenterIts}{T4transport}
}
\concept{dist}
