/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5HFpkg.h"     
#include "H5MFprivate.h" 
#include "H5VMprivate.h" 

static herr_t H5HF__iblock_pin(H5HF_indirect_t *iblock);
static herr_t H5HF__iblock_unpin(H5HF_indirect_t *iblock);
static herr_t H5HF__man_iblock_root_halve(H5HF_indirect_t *root_iblock);
static herr_t H5HF__man_iblock_root_revert(H5HF_indirect_t *root_iblock);

H5FL_DEFINE(H5HF_indirect_t);

H5FL_SEQ_DEFINE(H5HF_indirect_ent_t);

H5FL_SEQ_DEFINE(H5HF_indirect_filt_ent_t);

H5FL_SEQ_DEFINE(H5HF_indirect_ptr_t);

static herr_t
H5HF__iblock_pin(H5HF_indirect_t *iblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);

    
    if (H5AC_pin_protected_entry(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPIN, FAIL, "unable to pin fractal heap indirect block");

    
    if (iblock->parent) {
        H5HF_indirect_t *par_iblock = iblock->parent; 
        unsigned         indir_idx;                   

        
        assert(par_iblock->child_iblocks);
        assert(iblock->par_entry >=
               (iblock->hdr->man_dtable.max_direct_rows * iblock->hdr->man_dtable.cparam.width));

        
        indir_idx = iblock->par_entry -
                    (iblock->hdr->man_dtable.max_direct_rows * iblock->hdr->man_dtable.cparam.width);

        
        assert(par_iblock->child_iblocks[indir_idx] == NULL);
        par_iblock->child_iblocks[indir_idx] = iblock;
    } 
    else {
        
        if (iblock->block_off == 0) {
            
            assert(0 == (iblock->hdr->root_iblock_flags & H5HF_ROOT_IBLOCK_PINNED));

            
            if (0 == iblock->hdr->root_iblock_flags) {
                assert(NULL == iblock->hdr->root_iblock);
                iblock->hdr->root_iblock = iblock;
            } 

            
            iblock->hdr->root_iblock_flags |= H5HF_ROOT_IBLOCK_PINNED;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__iblock_unpin(H5HF_indirect_t *iblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);

    
    if (H5AC_unpin_entry(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPIN, FAIL, "unable to unpin fractal heap indirect block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__iblock_incr(H5HF_indirect_t *iblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);
    assert(iblock->block_off == 0 || iblock->parent);

    
    if (iblock->rc == 0)
        if (H5HF__iblock_pin(iblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPIN, FAIL, "unable to pin fractal heap indirect block");

    
    iblock->rc++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__iblock_decr(H5HF_indirect_t *iblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);

    
    iblock->rc--;

    
    if (iblock->rc == 0) {

        
        if (iblock->parent) {
            H5HF_indirect_t *par_iblock = iblock->parent; 
            unsigned         indir_idx;                   

            
            assert(par_iblock->child_iblocks);
            assert(iblock->par_entry >=
                   (iblock->hdr->man_dtable.max_direct_rows * iblock->hdr->man_dtable.cparam.width));

            
            indir_idx = iblock->par_entry -
                        (iblock->hdr->man_dtable.max_direct_rows * iblock->hdr->man_dtable.cparam.width);

            
            assert(par_iblock->child_iblocks[indir_idx]);
            par_iblock->child_iblocks[indir_idx] = NULL;
        } 
        else {
            
            if (iblock->block_off == 0) {
                
                assert(iblock->hdr->root_iblock_flags & H5HF_ROOT_IBLOCK_PINNED);

                
                if (H5HF_ROOT_IBLOCK_PINNED == iblock->hdr->root_iblock_flags) {
                    assert(NULL != iblock->hdr->root_iblock);
                    iblock->hdr->root_iblock = NULL;
                } 

                
                iblock->hdr->root_iblock_flags &= (unsigned)(~(H5HF_ROOT_IBLOCK_PINNED));
            } 
        }     

        
        if (!iblock->removed_from_cache) {
            
            if (H5HF__iblock_unpin(iblock) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPIN, FAIL, "unable to unpin fractal heap indirect block");
        } 
        else {
            
            if (H5HF__man_iblock_dest(iblock) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to destroy fractal heap indirect block");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__iblock_dirty(H5HF_indirect_t *iblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);

    
    if (H5AC_mark_entry_dirty(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTMARKDIRTY, FAIL, "unable to mark fractal heap indirect block as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_root_create(H5HF_hdr_t *hdr, size_t min_dblock_size)
{
    H5HF_indirect_t *iblock;              
    haddr_t          iblock_addr;         
    hsize_t          acc_dblock_free;     
    bool             have_direct_block;   
    bool             did_protect;         
    unsigned         nrows;               
    unsigned         u;                   
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (hdr->man_dtable.cparam.start_root_rows == 0)
        nrows = hdr->man_dtable.max_root_rows;
    else {
        unsigned rows_needed;   
        unsigned block_row_off; 

        nrows = hdr->man_dtable.cparam.start_root_rows;

        block_row_off = H5VM_log2_of2((uint32_t)min_dblock_size) -
                        H5VM_log2_of2((uint32_t)hdr->man_dtable.cparam.start_block_size);
        if (block_row_off > 0)
            block_row_off++; 
        rows_needed = 1 + block_row_off;
        if (nrows < rows_needed)
            nrows = rows_needed;
    } 

    
    if (H5HF__man_iblock_create(hdr, NULL, 0, nrows, hdr->man_dtable.max_root_rows, &iblock_addr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "can't allocate fractal heap indirect block");

    

    
    if (NULL == (iblock = H5HF__man_iblock_protect(hdr, iblock_addr, nrows, NULL, 0, false,
                                                   H5AC__NO_FLAGS_SET, &did_protect)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap indirect block");

    
    have_direct_block = H5_addr_defined(hdr->man_dtable.table_addr);
    if (have_direct_block) {
        H5HF_direct_t *dblock; 

        
        if (NULL == (dblock = H5HF__man_dblock_protect(hdr, hdr->man_dtable.table_addr,
                                                       hdr->man_dtable.cparam.start_block_size, NULL, 0,
                                                       H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap direct block");

        
        dblock->parent    = iblock;
        dblock->par_entry = 0;

        
        if (H5AC_destroy_flush_dependency(dblock->fd_parent, dblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");
        dblock->fd_parent = NULL;

        
        if (H5AC_create_flush_dependency(iblock, dblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDEPEND, FAIL, "unable to create flush dependency");
        dblock->fd_parent = iblock;

        if (H5HF__man_iblock_attach(iblock, 0, hdr->man_dtable.table_addr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTATTACH, FAIL,
                        "can't attach root direct block to parent indirect block");

        
        if (hdr->filter_len > 0) {
            
            iblock->filt_ents[0].size        = hdr->pline_root_direct_size;
            iblock->filt_ents[0].filter_mask = hdr->pline_root_direct_filter_mask;

            
            hdr->pline_root_direct_size        = 0;
            hdr->pline_root_direct_filter_mask = 0;
        } 

        
        if (H5HF__space_create_root(hdr, iblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTSET, FAIL,
                        "can't set free space section info to new root indirect block");

        
        if (H5AC_unprotect(hdr->f, H5AC_FHEAP_DBLOCK, hdr->man_dtable.table_addr, dblock,
                           H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap direct block");
        dblock = NULL;
    } 

    
    if (H5HF__hdr_start_iter(hdr, iblock,
                             (hsize_t)(have_direct_block ? hdr->man_dtable.cparam.start_block_size : 0),
                             have_direct_block) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize block iterator");

    
    if (min_dblock_size > hdr->man_dtable.cparam.start_block_size)
        
        if (H5HF__hdr_skip_blocks(hdr, iblock, have_direct_block,
                                  ((nrows - 1) * hdr->man_dtable.cparam.width) - have_direct_block) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't add skipped blocks to heap's free space");

    
    if (H5HF__iblock_dirty(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark indirect block as dirty");

    
    if (H5HF__man_iblock_unprotect(iblock, H5AC__DIRTIED_FLAG, did_protect) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");
    iblock = NULL;

    
    hdr->man_dtable.curr_root_rows = nrows;
    hdr->man_dtable.table_addr     = iblock_addr;

    
    acc_dblock_free = 0;
    for (u = 0; u < nrows; u++)
        acc_dblock_free += hdr->man_dtable.row_tot_dblock_free[u] * hdr->man_dtable.cparam.width;

    
    if (have_direct_block)
        acc_dblock_free -= hdr->man_dtable.row_tot_dblock_free[0];

    
    if (H5HF__hdr_adjust_heap(hdr, hdr->man_dtable.row_block_off[nrows], (hssize_t)acc_dblock_free) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTEXTEND, FAIL, "can't increase space to cover root direct block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_root_double(H5HF_hdr_t *hdr, size_t min_dblock_size)
{
    H5HF_indirect_t *iblock;          
    haddr_t          new_addr;        
    hsize_t          acc_dblock_free; 
    hsize_t          next_size;       
    hsize_t          old_iblock_size; 
    unsigned         next_row;        
    unsigned         next_entry;      
    unsigned         new_next_entry = 0; 
    unsigned         min_nrows      = 0; 
    unsigned         old_nrows;          
    unsigned         new_nrows;          
    bool             skip_direct_rows = false; 
    size_t           u;                        
    herr_t           ret_value = SUCCEED;      

    FUNC_ENTER_PACKAGE

    
    if (H5HF__man_iter_curr(&hdr->next_block, &next_row, NULL, &next_entry, &iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "unable to retrieve current block iterator location");
    next_size = hdr->man_dtable.row_block_size[next_row];

    
    assert(iblock->parent == NULL);
    assert(iblock->block_off == 0);

    
    old_nrows = iblock->nrows;

    
    if (iblock->nrows < hdr->man_dtable.max_direct_rows && min_dblock_size > next_size) {
        
        assert(min_dblock_size > hdr->man_dtable.cparam.start_block_size);

        
        skip_direct_rows = true;

        
        min_nrows = 1 + H5HF__dtable_size_to_row(&hdr->man_dtable, min_dblock_size);

        
        new_next_entry = (min_nrows - 1) * hdr->man_dtable.cparam.width;
    } 

    
    new_nrows = MAX(min_nrows, MIN(2 * iblock->nrows, iblock->max_rows));

    
    
    if (!H5F_IS_TMP_ADDR(hdr->f, iblock->addr))
        
        if (H5MF_xfree(hdr->f, H5FD_MEM_FHEAP_IBLOCK, iblock->addr, (hsize_t)iblock->size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL,
                        "unable to free fractal heap indirect block file space");

    
    iblock->nrows   = new_nrows;
    old_iblock_size = iblock->size;
    iblock->size    = H5HF_MAN_INDIRECT_SIZE(hdr, iblock->nrows);

    
    if (H5F_USE_TMP_SPACE(hdr->f)) {
        if (HADDR_UNDEF == (new_addr = H5MF_alloc_tmp(hdr->f, (hsize_t)iblock->size)))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL,
                        "file allocation failed for fractal heap indirect block");
    } 
    else {
        if (HADDR_UNDEF == (new_addr = H5MF_alloc(hdr->f, H5FD_MEM_FHEAP_IBLOCK, (hsize_t)iblock->size)))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL,
                        "file allocation failed for fractal heap indirect block");
    } 

    
    if (old_iblock_size != iblock->size) {
        if (H5AC_resize_entry(iblock, (size_t)iblock->size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize fractal heap indirect block");
    } 

    
    if (H5_addr_ne(iblock->addr, new_addr)) {
        if (H5AC_move_entry(hdr->f, H5AC_FHEAP_IBLOCK, iblock->addr, new_addr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTMOVE, FAIL, "unable to move fractal heap root indirect block");
        iblock->addr = new_addr;
    } 

    
    if (NULL == (iblock->ents = H5FL_SEQ_REALLOC(H5HF_indirect_ent_t, iblock->ents,
                                                 (size_t)(iblock->nrows * hdr->man_dtable.cparam.width))))
        HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "memory allocation failed for direct entries");

    
    if (skip_direct_rows)
        
        if (H5HF__hdr_skip_blocks(hdr, iblock, next_entry, (new_next_entry - next_entry)) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't add skipped blocks to heap's free space");

    
    acc_dblock_free = 0;
    for (u = (old_nrows * hdr->man_dtable.cparam.width); u < (iblock->nrows * hdr->man_dtable.cparam.width);
         u++) {
        unsigned row = (unsigned)(u / hdr->man_dtable.cparam.width); 

        iblock->ents[u].addr = HADDR_UNDEF;
        acc_dblock_free += hdr->man_dtable.row_tot_dblock_free[row];
    } 

    
    if (hdr->filter_len > 0 && old_nrows < hdr->man_dtable.max_direct_rows) {
        unsigned dir_rows; 

        
        dir_rows = MIN(iblock->nrows, hdr->man_dtable.max_direct_rows);
        assert(dir_rows > old_nrows);

        
        if (NULL == (iblock->filt_ents = H5FL_SEQ_REALLOC(H5HF_indirect_filt_ent_t, iblock->filt_ents,
                                                          (size_t)(dir_rows * hdr->man_dtable.cparam.width))))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "memory allocation failed for filtered direct entries");

        
        for (u = (old_nrows * hdr->man_dtable.cparam.width); u < (dir_rows * hdr->man_dtable.cparam.width);
             u++) {
            iblock->filt_ents[u].size        = 0;
            iblock->filt_ents[u].filter_mask = 0;
        } 
    }     

    
    if (iblock->nrows > hdr->man_dtable.max_direct_rows) {
        unsigned indir_rows;     
        unsigned old_indir_rows; 

        
        indir_rows = iblock->nrows - hdr->man_dtable.max_direct_rows;

        
        if (NULL ==
            (iblock->child_iblocks = H5FL_SEQ_REALLOC(H5HF_indirect_ptr_t, iblock->child_iblocks,
                                                      (size_t)(indir_rows * hdr->man_dtable.cparam.width))))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "memory allocation failed for filtered direct entries");

        
        if (old_nrows < hdr->man_dtable.max_direct_rows)
            old_indir_rows = 0;
        else
            old_indir_rows = old_nrows - hdr->man_dtable.max_direct_rows;

        
        for (u = (old_indir_rows * hdr->man_dtable.cparam.width);
             u < (indir_rows * hdr->man_dtable.cparam.width); u++)
            iblock->child_iblocks[u] = NULL;
    } 

    
    if (H5HF__iblock_dirty(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark indirect block as dirty");

    
    hdr->man_dtable.curr_root_rows = new_nrows;
    hdr->man_dtable.table_addr     = new_addr;

    
    if (H5HF__hdr_adjust_heap(hdr, 2 * hdr->man_dtable.row_block_off[new_nrows - 1],
                              (hssize_t)acc_dblock_free) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTEXTEND, FAIL, "can't increase space to cover root direct block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__man_iblock_root_halve(H5HF_indirect_t *iblock)
{
    H5HF_hdr_t *hdr = iblock->hdr;   
    haddr_t     new_addr;            
    hsize_t     acc_dblock_free;     
    hsize_t     old_size;            
    unsigned    max_child_row;       
    unsigned    old_nrows;           
    unsigned    new_nrows;           
    unsigned    u;                   
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);
    assert(iblock->parent == NULL);
    assert(hdr);

    
    max_child_row = iblock->max_child / hdr->man_dtable.cparam.width;

    
    new_nrows = (unsigned)1 << (1 + H5VM_log2_gen((uint64_t)max_child_row));

    
    
    if (!H5F_IS_TMP_ADDR(hdr->f, iblock->addr))
        
        if (H5MF_xfree(hdr->f, H5FD_MEM_FHEAP_IBLOCK, iblock->addr, (hsize_t)iblock->size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL,
                        "unable to free fractal heap indirect block file space");

    
    acc_dblock_free = 0;
    for (u = new_nrows; u < iblock->nrows; u++)
        acc_dblock_free += hdr->man_dtable.row_tot_dblock_free[u] * hdr->man_dtable.cparam.width;

    
    old_nrows     = iblock->nrows;
    iblock->nrows = new_nrows;
    old_size      = iblock->size;
    iblock->size  = H5HF_MAN_INDIRECT_SIZE(hdr, iblock->nrows);

    
    if (H5F_USE_TMP_SPACE(hdr->f)) {
        if (HADDR_UNDEF == (new_addr = H5MF_alloc_tmp(hdr->f, (hsize_t)iblock->size)))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL,
                        "file allocation failed for fractal heap indirect block");
    } 
    else {
        if (HADDR_UNDEF == (new_addr = H5MF_alloc(hdr->f, H5FD_MEM_FHEAP_IBLOCK, (hsize_t)iblock->size)))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL,
                        "file allocation failed for fractal heap indirect block");
    } 

    
    if (old_size != iblock->size) {
        if (H5AC_resize_entry(iblock, (size_t)iblock->size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize fractal heap indirect block");
    } 

    
    if (H5_addr_ne(iblock->addr, new_addr)) {
        if (H5AC_move_entry(hdr->f, H5AC_FHEAP_IBLOCK, iblock->addr, new_addr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTSPLIT, FAIL, "unable to move fractal heap root indirect block");
        iblock->addr = new_addr;
    } 

    
    if (NULL == (iblock->ents = H5FL_SEQ_REALLOC(H5HF_indirect_ent_t, iblock->ents,
                                                 (size_t)(iblock->nrows * hdr->man_dtable.cparam.width))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for direct entries");

    
    if (hdr->filter_len > 0 && new_nrows < hdr->man_dtable.max_direct_rows) {
        
        if (NULL ==
            (iblock->filt_ents = H5FL_SEQ_REALLOC(H5HF_indirect_filt_ent_t, iblock->filt_ents,
                                                  (size_t)(iblock->nrows * hdr->man_dtable.cparam.width))))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "memory allocation failed for filtered direct entries");
    } 

    
    if (old_nrows > hdr->man_dtable.max_direct_rows) {
        
        if (iblock->nrows > hdr->man_dtable.max_direct_rows) {
            unsigned indir_rows; 

            
            indir_rows = iblock->nrows - hdr->man_dtable.max_direct_rows;

            
            if (NULL == (iblock->child_iblocks =
                             H5FL_SEQ_REALLOC(H5HF_indirect_ptr_t, iblock->child_iblocks,
                                              (size_t)(indir_rows * hdr->man_dtable.cparam.width))))
                HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL,
                            "memory allocation failed for filtered direct entries");
        } 
        else
            iblock->child_iblocks =
                (H5HF_indirect_ptr_t *)H5FL_SEQ_FREE(H5HF_indirect_ptr_t, iblock->child_iblocks);
    } 

    
    if (H5HF__iblock_dirty(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark indirect block as dirty");

    
    hdr->man_dtable.curr_root_rows = new_nrows;
    hdr->man_dtable.table_addr     = new_addr;

    
    if (H5HF__hdr_adjust_heap(hdr, 2 * hdr->man_dtable.row_block_off[new_nrows - 1],
                              -(hssize_t)acc_dblock_free) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't reduce space to cover root direct block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__man_iblock_root_revert(H5HF_indirect_t *root_iblock)
{
    H5HF_hdr_t    *hdr;                 
    H5HF_direct_t *dblock = NULL;       
    haddr_t        dblock_addr;         
    size_t         dblock_size;         
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(root_iblock);

    
    hdr         = root_iblock->hdr;
    dblock_addr = root_iblock->ents[0].addr;
    dblock_size = hdr->man_dtable.cparam.start_block_size;

    
    if (NULL == (dblock = H5HF__man_dblock_protect(hdr, dblock_addr, dblock_size, root_iblock, 0,
                                                   H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap direct block");
    assert(dblock->parent == root_iblock);
    assert(dblock->par_entry == 0);

    
    if (hdr->filter_len > 0) {
        
        hdr->pline_root_direct_size        = root_iblock->filt_ents[0].size;
        hdr->pline_root_direct_filter_mask = root_iblock->filt_ents[0].filter_mask;
    } 

    
    if (H5AC_destroy_flush_dependency(dblock->fd_parent, dblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");
    dblock->fd_parent = NULL;

    
    if (H5HF__man_iblock_detach(dblock->parent, 0) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTATTACH, FAIL, "can't detach direct block from parent indirect block");
    dblock->parent    = NULL;
    dblock->par_entry = 0;

    
    if (H5AC_create_flush_dependency(hdr, dblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEPEND, FAIL, "unable to create flush dependency");
    dblock->fd_parent = hdr;

    
    hdr->man_dtable.curr_root_rows = 0;
    hdr->man_dtable.table_addr     = dblock_addr;

    
    if (H5HF__hdr_reset_iter(hdr, (hsize_t)dblock_size) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't reset block iterator");

    
    if (H5HF__hdr_adjust_heap(hdr, (hsize_t)hdr->man_dtable.cparam.start_block_size,
                              (hssize_t)hdr->man_dtable.row_tot_dblock_free[0]) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTEXTEND, FAIL, "can't increase space to cover root direct block");

    
    if (H5HF__space_revert_root(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRESET, FAIL, "can't reset free space section info");

done:
    if (dblock && H5AC_unprotect(hdr->f, H5AC_FHEAP_DBLOCK, dblock_addr, dblock, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap direct block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_alloc_row(H5HF_hdr_t *hdr, H5HF_free_section_t **sec_node)
{
    H5HF_indirect_t     *iblock       = NULL;      
    H5HF_free_section_t *old_sec_node = *sec_node; 
    unsigned             dblock_entry;             
    bool                 iblock_held = false;      
    herr_t               ret_value   = SUCCEED;    

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sec_node && old_sec_node);
    assert(old_sec_node->u.row.row < hdr->man_dtable.max_direct_rows);

    
    if (old_sec_node->sect_info.state == H5FS_SECT_SERIALIZED ||
        (H5FS_SECT_SERIALIZED == old_sec_node->u.row.under->sect_info.state) ||
        (true == old_sec_node->u.row.under->u.indirect.u.iblock->removed_from_cache))
        
        if (H5HF__sect_row_revive(hdr, old_sec_node) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTREVIVE, FAIL, "can't revive indirect section");

    
    if (NULL == (iblock = H5HF__sect_row_get_iblock(old_sec_node)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve indirect block for row section");

    
    if (H5HF__iblock_incr(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't increment reference count on shared indirect block");
    iblock_held = true;

    
    if (H5HF__sect_row_reduce(hdr, old_sec_node, &dblock_entry) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't reduce row section node");

    
    if (H5HF__man_dblock_create(hdr, iblock, dblock_entry, NULL, sec_node) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "can't allocate fractal heap direct block");

done:
    
    if (iblock_held)
        if (H5HF__iblock_decr(iblock) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared indirect block")

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_create(H5HF_hdr_t *hdr, H5HF_indirect_t *par_iblock, unsigned par_entry, unsigned nrows,
                        unsigned max_rows, haddr_t *addr_p)
{
    H5HF_indirect_t *iblock = NULL;       
    size_t           u;                   
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(nrows > 0);
    assert(addr_p);

    
    if (NULL == (iblock = H5FL_MALLOC(H5HF_indirect_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                    "memory allocation failed for fractal heap indirect block");

    
    memset(&iblock->cache_info, 0, sizeof(H5AC_info_t));

    
    iblock->hdr = hdr;
    if (H5HF__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't increment reference count on shared heap header");

    
    iblock->rc                 = 0;
    iblock->nrows              = nrows;
    iblock->max_rows           = max_rows;
    iblock->removed_from_cache = false;

    
    iblock->size = H5HF_MAN_INDIRECT_SIZE(hdr, iblock->nrows);

    
    if (NULL == (iblock->ents = H5FL_SEQ_MALLOC(H5HF_indirect_ent_t,
                                                (size_t)(iblock->nrows * hdr->man_dtable.cparam.width))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for block entries");

    
    for (u = 0; u < (iblock->nrows * hdr->man_dtable.cparam.width); u++)
        iblock->ents[u].addr = HADDR_UNDEF;

    
    if (hdr->filter_len > 0) {
        unsigned dir_rows; 

        
        dir_rows = MIN(iblock->nrows, hdr->man_dtable.max_direct_rows);

        
        if (NULL == (iblock->filt_ents = H5FL_SEQ_CALLOC(H5HF_indirect_filt_ent_t,
                                                         (size_t)(dir_rows * hdr->man_dtable.cparam.width))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for block entries");
    } 
    else
        iblock->filt_ents = NULL;

    
    if (iblock->nrows > hdr->man_dtable.max_direct_rows) {
        unsigned indir_rows; 

        
        indir_rows = iblock->nrows - hdr->man_dtable.max_direct_rows;

        
        if (NULL == (iblock->child_iblocks = H5FL_SEQ_CALLOC(
                         H5HF_indirect_ptr_t, (size_t)(indir_rows * hdr->man_dtable.cparam.width))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for block entries");
    } 
    else
        iblock->child_iblocks = NULL;

    
    if (H5F_USE_TMP_SPACE(hdr->f)) {
        if (HADDR_UNDEF == (*addr_p = H5MF_alloc_tmp(hdr->f, (hsize_t)iblock->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                        "file allocation failed for fractal heap indirect block");
    } 
    else {
        if (HADDR_UNDEF == (*addr_p = H5MF_alloc(hdr->f, H5FD_MEM_FHEAP_IBLOCK, (hsize_t)iblock->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                        "file allocation failed for fractal heap indirect block");
    } 
    iblock->addr = *addr_p;

    
    iblock->parent    = par_iblock;
    iblock->par_entry = par_entry;
    if (iblock->parent) {
        
        if (H5HF__man_iblock_attach(iblock->parent, par_entry, *addr_p) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTATTACH, FAIL,
                        "can't attach indirect block to parent indirect block");

        
        
        iblock->block_off = par_iblock->block_off;
        iblock->block_off += hdr->man_dtable.row_block_off[par_entry / hdr->man_dtable.cparam.width];
        iblock->block_off += hdr->man_dtable.row_block_size[par_entry / hdr->man_dtable.cparam.width] *
                             (par_entry % hdr->man_dtable.cparam.width);

        
        iblock->fd_parent = par_iblock;
    } 
    else {
        iblock->block_off = 0; 

        
        iblock->fd_parent = hdr;
    } 

    
    iblock->nchildren = 0;
    iblock->max_child = 0;

    
    if (H5AC_insert_entry(hdr->f, H5AC_FHEAP_IBLOCK, *addr_p, iblock, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't add fractal heap indirect block to cache");

done:
    if (ret_value < 0)
        if (iblock)
            if (H5HF__man_iblock_dest(iblock) < 0)
                HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to destroy fractal heap indirect block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5HF_indirect_t *
H5HF__man_iblock_protect(H5HF_hdr_t *hdr, haddr_t iblock_addr, unsigned iblock_nrows,
                         H5HF_indirect_t *par_iblock, unsigned par_entry, bool must_protect, unsigned flags,
                         bool *did_protect)
{
    H5HF_parent_t    par_info;               
    H5HF_indirect_t *iblock         = NULL;  
    bool             should_protect = false; 
    H5HF_indirect_t *ret_value      = NULL;  

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(iblock_addr));
    assert(iblock_nrows > 0);
    assert(did_protect);

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    if (!must_protect) {
        
        if (par_iblock) {
            unsigned indir_idx; 

            
            assert(par_iblock->child_iblocks);
            assert(par_entry >= (hdr->man_dtable.max_direct_rows * hdr->man_dtable.cparam.width));

            
            indir_idx = par_entry - (hdr->man_dtable.max_direct_rows * hdr->man_dtable.cparam.width);

            
            if (par_iblock->child_iblocks[indir_idx])
                iblock = par_iblock->child_iblocks[indir_idx];
            else
                should_protect = true;
        } 
        else {
            
            if (H5_addr_eq(iblock_addr, hdr->man_dtable.table_addr)) {
                
                if (H5HF_ROOT_IBLOCK_PINNED == hdr->root_iblock_flags) {
                    
                    assert(NULL != hdr->root_iblock);

                    
                    iblock = hdr->root_iblock;
                } 
                else {
                    
                    assert(NULL == hdr->root_iblock);

                    should_protect = true;
                } 
            }     
            else
                should_protect = true;
        } 
    }     

    
    if (must_protect || should_protect) {
        H5HF_iblock_cache_ud_t cache_udata; 

        
        par_info.hdr    = hdr;
        par_info.iblock = par_iblock;
        par_info.entry  = par_entry;

        
        cache_udata.f        = hdr->f;
        cache_udata.par_info = &par_info;
        cache_udata.nrows    = &iblock_nrows;

        
        if (NULL == (iblock = (H5HF_indirect_t *)H5AC_protect(hdr->f, H5AC_FHEAP_IBLOCK, iblock_addr,
                                                              &cache_udata, flags)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, NULL, "unable to protect fractal heap indirect block");

        
        iblock->addr = iblock_addr;

        
        if (iblock->block_off == 0) {
            
            assert(0 == (hdr->root_iblock_flags & H5HF_ROOT_IBLOCK_PROTECTED));

            
            if (0 == hdr->root_iblock_flags) {
                assert(NULL == hdr->root_iblock);
                hdr->root_iblock = iblock;
            } 

            
            hdr->root_iblock_flags |= H5HF_ROOT_IBLOCK_PROTECTED;
        } 

        
        *did_protect = true;
    } 
    else
        
        *did_protect = false;

    
    ret_value = iblock;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_unprotect(H5HF_indirect_t *iblock, unsigned cache_flags, bool did_protect)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);

    
    
    if (did_protect) {
        
        if (iblock->block_off == 0) {
            
            assert(iblock->hdr->root_iblock_flags & H5HF_ROOT_IBLOCK_PROTECTED);

            
            if (H5HF_ROOT_IBLOCK_PROTECTED == iblock->hdr->root_iblock_flags) {
                assert(NULL != iblock->hdr->root_iblock);
                iblock->hdr->root_iblock = NULL;
            } 

            
            iblock->hdr->root_iblock_flags &= (unsigned)(~(H5HF_ROOT_IBLOCK_PROTECTED));
        } 

        
        if (H5AC_unprotect(iblock->hdr->f, H5AC_FHEAP_IBLOCK, iblock->addr, iblock, cache_flags) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_attach(H5HF_indirect_t *iblock, unsigned entry, haddr_t child_addr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);
    assert(H5_addr_defined(child_addr));
    assert(!H5_addr_defined(iblock->ents[entry].addr));

    
    if (H5HF__iblock_incr(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't increment reference count on shared indirect block");

    
    iblock->ents[entry].addr = child_addr;

    
    if (iblock->hdr->filter_len > 0) {
        unsigned row; 

        
        assert(iblock->filt_ents);

        
        row = entry / iblock->hdr->man_dtable.cparam.width;

        
        if (row < iblock->hdr->man_dtable.max_direct_rows)
            iblock->filt_ents[entry].size = iblock->hdr->man_dtable.row_block_size[row];
    } 

    
    if (entry > iblock->max_child)
        iblock->max_child = entry;

    
    iblock->nchildren++;

    
    if (H5HF__iblock_dirty(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark indirect block as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_detach(H5HF_indirect_t *iblock, unsigned entry)
{
    H5HF_hdr_t      *hdr;                 
    H5HF_indirect_t *del_iblock = NULL;   
    unsigned         row;                 
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);
    assert(iblock->nchildren);

    
    hdr = iblock->hdr;

    
    iblock->ents[entry].addr = HADDR_UNDEF;

    
    row = entry / hdr->man_dtable.cparam.width;

    
    if (hdr->filter_len > 0) {
        
        assert(iblock->filt_ents);

        
        if (row < hdr->man_dtable.max_direct_rows) {
            iblock->filt_ents[entry].size        = 0;
            iblock->filt_ents[entry].filter_mask = 0;
        } 
    }     

    
    if (row >= hdr->man_dtable.max_direct_rows) {
        unsigned indir_idx; 

        
        assert(iblock->child_iblocks);

        
        indir_idx = entry - (hdr->man_dtable.max_direct_rows * hdr->man_dtable.cparam.width);

        
        assert(iblock->child_iblocks[indir_idx]);

        
        iblock->child_iblocks[indir_idx] = NULL;
    } 

    
    
    iblock->nchildren--;

    
    if (entry == iblock->max_child) {
        if (iblock->nchildren > 0)
            while (!H5_addr_defined(iblock->ents[iblock->max_child].addr))
                iblock->max_child--;
        else
            iblock->max_child = 0;
    } 

    
    if (iblock->block_off == 0) {
        
        if (iblock->nchildren == 1 && H5_addr_defined(iblock->ents[0].addr))
            if (H5HF__man_iblock_root_revert(iblock) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL,
                            "can't convert root indirect block back to root direct block");

        
        if (!iblock->removed_from_cache) {
            
            if (iblock->nchildren > 0 && hdr->man_dtable.cparam.start_root_rows != 0 &&
                entry > iblock->max_child) {
                unsigned max_child_row; 

                
                max_child_row = iblock->max_child / hdr->man_dtable.cparam.width;

                
                if (iblock->nrows > 1 && max_child_row <= (iblock->nrows / 2))
                    if (H5HF__man_iblock_root_halve(iblock) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL,
                                    "can't reduce size of root indirect block");
            } 
        }     
    }         

    
    if (!iblock->removed_from_cache) {
        
        if (H5HF__iblock_dirty(iblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark indirect block as dirty");

        
        if (iblock->nchildren == 0) {
            bool did_protect = false; 

            
            
            if (NULL == (del_iblock = H5HF__man_iblock_protect(hdr, iblock->addr, iblock->nrows,
                                                               iblock->parent, iblock->par_entry, true,
                                                               H5AC__NO_FLAGS_SET, &did_protect)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap indirect block");
            assert(did_protect == true);

            
            if (iblock->block_off == 0 && hdr->man_dtable.curr_root_rows > 0)
                
                if (H5HF__hdr_empty(hdr) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't make heap empty");

            
            if (iblock->parent) {
                
                if (H5AC_destroy_flush_dependency(iblock->fd_parent, iblock) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");
                iblock->fd_parent = NULL;

                
                if (H5HF__man_iblock_detach(iblock->parent, iblock->par_entry) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTATTACH, FAIL, "can't detach from parent indirect block");
                iblock->parent    = NULL;
                iblock->par_entry = 0;
            } 
        }     
    }         

    
    
    if (H5HF__iblock_decr(iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement reference count on shared indirect block");
    iblock = NULL;

    
    if (del_iblock) {
        unsigned cache_flags    = H5AC__NO_FLAGS_SET; 
        bool     took_ownership = false; 

        
        if (del_iblock->rc > 0) {
            cache_flags |= (H5AC__DELETED_FLAG | H5AC__TAKE_OWNERSHIP_FLAG);
            cache_flags |= H5AC__UNPIN_ENTRY_FLAG;
            took_ownership = true;
        } 
        else {
            
            cache_flags |= H5AC__DELETED_FLAG;

            
            if (!H5F_IS_TMP_ADDR(hdr->f, del_iblock->addr))
                cache_flags |= H5AC__FREE_FILE_SPACE_FLAG;
        } 

        
        if (H5HF__man_iblock_unprotect(del_iblock, cache_flags, true) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");

        
        if (took_ownership) {
            
            if (!H5F_IS_TMP_ADDR(hdr->f, del_iblock->addr))
                if (H5MF_xfree(hdr->f, H5FD_MEM_FHEAP_IBLOCK, del_iblock->addr, (hsize_t)del_iblock->size) <
                    0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL,
                                "unable to free fractal heap indirect block file space");
            del_iblock->addr = HADDR_UNDEF;

            
            del_iblock->removed_from_cache = true;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_entry_addr(H5HF_indirect_t *iblock, unsigned entry, haddr_t *child_addr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iblock);
    assert(child_addr);

    
    *child_addr = iblock->ents[entry].addr;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__man_iblock_delete(H5HF_hdr_t *hdr, haddr_t iblock_addr, unsigned iblock_nrows,
                        H5HF_indirect_t *par_iblock, unsigned par_entry)
{
    H5HF_indirect_t *iblock;                           
    unsigned         row, col;                         
    unsigned         entry;                            
    unsigned         cache_flags = H5AC__NO_FLAGS_SET; 
    bool             did_protect;                      
    herr_t           ret_value = SUCCEED;              

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(iblock_addr));
    assert(iblock_nrows > 0);

    
    if (NULL == (iblock = H5HF__man_iblock_protect(hdr, iblock_addr, iblock_nrows, par_iblock, par_entry,
                                                   true, H5AC__NO_FLAGS_SET, &did_protect)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap indirect block");
    assert(iblock->nchildren > 0);
    assert(did_protect == true);

    
    entry = 0;
    for (row = 0; row < iblock->nrows; row++) {
        
        for (col = 0; col < hdr->man_dtable.cparam.width; col++, entry++) {
            
            if (H5_addr_defined(iblock->ents[entry].addr)) {
                
                if (row < hdr->man_dtable.max_direct_rows) {
                    hsize_t dblock_size; 

                    
                    if (hdr->filter_len > 0)
                        dblock_size = iblock->filt_ents[entry].size;
                    else
                        dblock_size = hdr->man_dtable.row_block_size[row];

                    
                    if (H5HF__man_dblock_delete(hdr->f, iblock->ents[entry].addr, dblock_size) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL,
                                    "unable to release fractal heap child direct block");
                } 
                else {
                    hsize_t  row_block_size; 
                    unsigned child_nrows;    

                    
                    row_block_size = (hsize_t)hdr->man_dtable.row_block_size[row];

                    
                    child_nrows = H5HF__dtable_size_to_rows(&hdr->man_dtable, row_block_size);

                    
                    if (H5HF__man_iblock_delete(hdr, iblock->ents[entry].addr, child_nrows, iblock, entry) <
                        0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL,
                                    "unable to release fractal heap child indirect block");
                } 
            }     
        }         
    }             

#ifndef NDEBUG
    {
        unsigned iblock_status = 0; 

        
        if (H5AC_get_entry_status(hdr->f, iblock_addr, &iblock_status) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                        "unable to check metadata cache status for indirect block");

        
        assert(!(iblock_status & H5AC_ES__IS_PINNED));
    }
#endif 

    
    cache_flags |= H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG;

    
    if (!H5F_IS_TMP_ADDR(hdr->f, iblock_addr))
        cache_flags |= H5AC__FREE_FILE_SPACE_FLAG;

done:
    
    if (iblock && H5HF__man_iblock_unprotect(iblock, cache_flags, did_protect) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_size(H5F_t *f, H5HF_hdr_t *hdr, haddr_t iblock_addr, unsigned nrows,
                      H5HF_indirect_t *par_iblock, unsigned par_entry, hsize_t *heap_size)
{
    H5HF_indirect_t *iblock = NULL;       
    bool             did_protect;         
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(hdr);
    assert(H5_addr_defined(iblock_addr));
    assert(heap_size);

    
    if (NULL == (iblock = H5HF__man_iblock_protect(hdr, iblock_addr, nrows, par_iblock, par_entry, false,
                                                   H5AC__READ_ONLY_FLAG, &did_protect)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTLOAD, FAIL, "unable to load fractal heap indirect block");

    
    *heap_size += iblock->size;

    
    if (iblock->nrows > hdr->man_dtable.max_direct_rows) {
        unsigned first_row_bits;    
        unsigned num_indirect_rows; 
        unsigned entry;             
        size_t   u;                 

        entry          = hdr->man_dtable.max_direct_rows * hdr->man_dtable.cparam.width;
        first_row_bits = H5VM_log2_of2((uint32_t)hdr->man_dtable.cparam.start_block_size) +
                         H5VM_log2_of2(hdr->man_dtable.cparam.width);
        num_indirect_rows = (H5VM_log2_gen(hdr->man_dtable.row_block_size[hdr->man_dtable.max_direct_rows]) -
                             first_row_bits) +
                            1;
        for (u = hdr->man_dtable.max_direct_rows; u < iblock->nrows; u++, num_indirect_rows++) {
            size_t v; 

            for (v = 0; v < hdr->man_dtable.cparam.width; v++, entry++)
                if (H5_addr_defined(iblock->ents[entry].addr))
                    if (H5HF__man_iblock_size(f, hdr, iblock->ents[entry].addr, num_indirect_rows, iblock,
                                              entry, heap_size) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTLOAD, FAIL,
                                    "unable to get fractal heap storage info for indirect block");
        } 
    }     

done:
    
    if (iblock && H5HF__man_iblock_unprotect(iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");
    iblock = NULL;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_parent_info(const H5HF_hdr_t *hdr, hsize_t block_off, hsize_t *ret_par_block_off,
                             unsigned *ret_entry)
{
    hsize_t  par_block_off;              
    hsize_t  prev_par_block_off;         
    unsigned row, col;                   
    unsigned prev_row = 0, prev_col = 0; 
    herr_t   ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(block_off > 0);
    assert(ret_entry);

    
    if (H5HF__dtable_lookup(&hdr->man_dtable, block_off, &row, &col) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPUTE, FAIL, "can't compute row & column of block");

    
    assert(row >= hdr->man_dtable.max_direct_rows);

    
    prev_par_block_off = par_block_off = 0;
    while (row >= hdr->man_dtable.max_direct_rows) {
        
        prev_par_block_off = par_block_off;

        
        
        par_block_off += hdr->man_dtable.row_block_off[row];
        par_block_off += hdr->man_dtable.row_block_size[row] * col;

        
        prev_row = row;
        prev_col = col;

        
        if (H5HF__dtable_lookup(&hdr->man_dtable, (block_off - par_block_off), &row, &col) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPUTE, FAIL, "can't compute row & column of block");
    } 

    
    assert(row == 0);
    assert(col == 0);

    
    *ret_par_block_off = prev_par_block_off;
    *ret_entry         = (prev_row * hdr->man_dtable.cparam.width) + prev_col;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iblock_dest(H5HF_indirect_t *iblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iblock);
    assert(iblock->rc == 0);

    
    assert(iblock->hdr);
    if (H5HF__hdr_decr(iblock->hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement reference count on shared heap header");
    if (iblock->parent)
        if (H5HF__iblock_decr(iblock->parent) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared indirect block");

    
    if (iblock->ents)
        iblock->ents = H5FL_SEQ_FREE(H5HF_indirect_ent_t, iblock->ents);
    if (iblock->filt_ents)
        iblock->filt_ents = H5FL_SEQ_FREE(H5HF_indirect_filt_ent_t, iblock->filt_ents);
    if (iblock->child_iblocks)
        iblock->child_iblocks = H5FL_SEQ_FREE(H5HF_indirect_ptr_t, iblock->child_iblocks);

    
    iblock = H5FL_FREE(H5HF_indirect_t, iblock);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
