/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Zmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  
#include "H5Tprivate.h"  
#include "H5Zpkg.h"      

static herr_t H5Z__set_local_shuffle(hid_t dcpl_id, hid_t type_id, hid_t space_id);
static size_t H5Z__filter_shuffle(unsigned flags, size_t cd_nelmts, const unsigned cd_values[], size_t nbytes,
                                  size_t *buf_size, void **buf);

const H5Z_class2_t H5Z_SHUFFLE[1] = {{
    H5Z_CLASS_T_VERS,       
    H5Z_FILTER_SHUFFLE,     
    1,                      
    1,                      
    "shuffle",              
    NULL,                   
    H5Z__set_local_shuffle, 
    H5Z__filter_shuffle,    
}};

#define H5Z_SHUFFLE_PARM_SIZE 0 

static herr_t
H5Z__set_local_shuffle(hid_t dcpl_id, hid_t type_id, hid_t H5_ATTR_UNUSED space_id)
{
    H5P_genplist_t *dcpl_plist;                          
    const H5T_t    *type;                                
    unsigned        flags;                               
    size_t          cd_nelmts = H5Z_SHUFFLE_USER_NPARMS; 
    unsigned        cd_values[H5Z_SHUFFLE_TOTAL_NPARMS]; 
    herr_t          ret_value = SUCCEED;                 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (dcpl_plist = H5P_object_verify(dcpl_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (NULL == (type = (const H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");

    
    if (H5P_get_filter_by_id(dcpl_plist, H5Z_FILTER_SHUFFLE, &flags, &cd_nelmts, cd_values, (size_t)0, NULL,
                             NULL) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, FAIL, "can't get shuffle parameters");

    
    if ((cd_values[H5Z_SHUFFLE_PARM_SIZE] = (unsigned)H5T_get_size(type)) == 0)
        HGOTO_ERROR(H5E_PLINE, H5E_BADTYPE, FAIL, "bad datatype size");

    
    if (H5P_modify_filter(dcpl_plist, H5Z_FILTER_SHUFFLE, flags, (size_t)H5Z_SHUFFLE_TOTAL_NPARMS,
                          cd_values) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTSET, FAIL, "can't set local shuffle parameters");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5Z__filter_shuffle(unsigned flags, size_t cd_nelmts, const unsigned cd_values[], size_t nbytes,
                    size_t *buf_size, void **buf)
{
    void          *dest  = NULL;  
    unsigned char *_src  = NULL;  
    unsigned char *_dest = NULL;  
    unsigned       bytesoftype;   
    size_t         numofelements; 
    size_t         i;             
#ifdef NO_DUFFS_DEVICE
    size_t j;             
#endif                    
    size_t leftover;      
    size_t ret_value = 0; 

    FUNC_ENTER_PACKAGE

    
    if (cd_nelmts != H5Z_SHUFFLE_TOTAL_NPARMS || cd_values[H5Z_SHUFFLE_PARM_SIZE] == 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, 0, "invalid shuffle parameters");

    
    bytesoftype = cd_values[H5Z_SHUFFLE_PARM_SIZE];

    
    numofelements = nbytes / bytesoftype;

    
    if (bytesoftype > 1 && numofelements > 1) {
        
        leftover = nbytes % bytesoftype;

        
        if (NULL == (dest = H5MM_malloc(nbytes)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, 0, "memory allocation failed for shuffle buffer");

        if (flags & H5Z_FLAG_REVERSE) {
            
            _src = (unsigned char *)(*buf);

            
            for (i = 0; i < bytesoftype; i++) {
                _dest = ((unsigned char *)dest) + i;
#define DUFF_GUTS                                                                                            \
    *_dest = *_src++;                                                                                        \
    _dest += bytesoftype;
#ifdef NO_DUFFS_DEVICE
                j = numofelements;
                while (j > 0) {
                    DUFF_GUTS;

                    j--;
                } 
#else             
                {
                    size_t duffs_index; 

                    duffs_index = (numofelements + 7) / 8;
                    switch (numofelements % 8) {
                        default:
                            assert(0 && "This Should never be executed!");
                            break;
                        case 0:
                            do {
                                DUFF_GUTS
                                
                                H5_ATTR_FALLTHROUGH
                                case 7:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 6:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 5:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 4:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 3:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 2:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 1:
                                    DUFF_GUTS
                            } while (--duffs_index > 0);
                    } 
                }
#endif            
#undef DUFF_GUTS
            } 

            
            if (leftover > 0) {
                
                _dest -= (bytesoftype - 1);
                H5MM_memcpy((void *)_dest, (void *)_src, leftover);
            }
        } 
        else {
            
            _dest = (unsigned char *)dest;

            
            for (i = 0; i < bytesoftype; i++) {
                _src = ((unsigned char *)(*buf)) + i;
#define DUFF_GUTS                                                                                            \
    *_dest++ = *_src;                                                                                        \
    _src += bytesoftype;
#ifdef NO_DUFFS_DEVICE
                j = numofelements;
                while (j > 0) {
                    DUFF_GUTS;

                    j--;
                } 
#else             
                {
                    size_t duffs_index; 

                    duffs_index = (numofelements + 7) / 8;
                    switch (numofelements % 8) {
                        default:
                            assert(0 && "This Should never be executed!");
                            break;
                        case 0:
                            do {
                                DUFF_GUTS
                                
                                H5_ATTR_FALLTHROUGH
                                case 7:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 6:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 5:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 4:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 3:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 2:
                                    DUFF_GUTS
                                    
                                    H5_ATTR_FALLTHROUGH
                                case 1:
                                    DUFF_GUTS
                            } while (--duffs_index > 0);
                    } 
                }
#endif            
#undef DUFF_GUTS
            } 

            
            if (leftover > 0) {
                
                _src -= (bytesoftype - 1);
                H5MM_memcpy((void *)_dest, (void *)_src, leftover);
            }
        } 

        
        H5MM_xfree(*buf);

        
        *buf      = dest;
        *buf_size = nbytes;
    } 

    
    ret_value = nbytes;

done:
    FUNC_LEAVE_NOAPI(ret_value)
}
