#ifndef DataStruct_H
#define DataStruct_H

#include <vector>
#include <cstdint>
#include <limits>
#include <string>
#include <utility> // for std::move

struct PartialCorRes {
  int first;
  double second;
  double third;

  /// Default constructor: initializes all members to zero.
  PartialCorRes() : first(0), second(0.0), third(0.0) {}

  /// Parameterized constructor: initializes all members.
  PartialCorRes(int f, double s, double t) : first(f), second(s), third(t) {}

  /// Defaulted copy/move constructors and assignment operators.
  PartialCorRes(const PartialCorRes&) = default;
  PartialCorRes& operator=(const PartialCorRes&) = default;
  PartialCorRes(PartialCorRes&&) noexcept = default;
  PartialCorRes& operator=(PartialCorRes&&) noexcept = default;
};

struct CMCRes {
  std::vector<double> cross_mapping;                 ///< Cross mapping values.
  std::vector<std::vector<double>> causal_strength;  ///< Causal strength matrix.

  /// Default constructor.
  CMCRes() = default;

  /// Copy constructor (for lvalues).
  CMCRes(const std::vector<double>& cross_mapping,
         const std::vector<std::vector<double>>& causal_strength)
    : cross_mapping(cross_mapping), causal_strength(causal_strength) {}

  /// Move constructor (for rvalues).
  CMCRes(std::vector<double>&& cross_mapping,
         std::vector<std::vector<double>>&& causal_strength) noexcept
    : cross_mapping(std::move(cross_mapping)), causal_strength(std::move(causal_strength)) {}

  /// Defaulted copy/move assignment operators.
  CMCRes(const CMCRes&) = default;
  CMCRes& operator=(const CMCRes&) = default;
  CMCRes(CMCRes&&) noexcept = default;
  CMCRes& operator=(CMCRes&&) noexcept = default;
};

/**
 * @brief Represents the result of an intersection calculation.
 *
 * This structure stores:
 * - `libsize`: The library size.
 * - `Intersection`: The intersection values as a vector of doubles.
 *
 */
struct IntersectionRes {
  size_t libsize;                   ///< Library size.
  std::vector<double> Intersection; ///< Intersection values.

  /// Default constructor: initializes libsize to 0 and empty Intersection vector.
  IntersectionRes() : libsize(0), Intersection() {}

  /// Copy constructor (for lvalues): deep copies Intersection vector.
  IntersectionRes(size_t t, const std::vector<double>& x)
    : libsize(t), Intersection(x) {}

  /// Move constructor (for rvalues): moves Intersection vector.
  IntersectionRes(size_t t, std::vector<double>&& x) noexcept
    : libsize(t), Intersection(std::move(x)) {}

  /// Defaulted copy/move assignment operators.
  IntersectionRes(const IntersectionRes&) = default;
  IntersectionRes& operator=(const IntersectionRes&) = default;
  IntersectionRes(IntersectionRes&&) noexcept = default;
  IntersectionRes& operator=(IntersectionRes&&) noexcept = default;
};

/**
 * @brief Holds the results of DeLong placement calculations.
 *
 * This structure stores:
 * - `theta`: The AUC or placement statistic.
 * - `X`: Placement values for the positive class.
 * - `Y`: Placement values for the negative class.
 *
 * The move constructor and move assignment are marked `noexcept`
 * to enable efficient transfers when stored in standard containers
 * like `std::vector`.
 */
struct DeLongPlacementsRes {
  double theta;                  ///< The AUC or placement statistic.
  std::vector<double> X;         ///< Placement values for the positive class.
  std::vector<double> Y;         ///< Placement values for the negative class.

  /**
   * @brief Default constructor.
   * Initializes theta = 0.0 and empty vectors.
   */
  DeLongPlacementsRes() : theta(0.0), X(), Y() {}

  /**
   * @brief Copy constructor (for lvalues).
   * Performs deep copies of input vectors.
   *
   * @param t The statistic value (e.g., theta).
   * @param x The placement values for the positive class.
   * @param y The placement values for the negative class.
   */
  DeLongPlacementsRes(double t, const std::vector<double>& x, const std::vector<double>& y)
    : theta(t), X(x), Y(y) {}

  /**
   * @brief Move constructor (for rvalues).
   * Transfers ownership of vectors without copying.
   */
  DeLongPlacementsRes(double t, std::vector<double>&& x, std::vector<double>&& y) noexcept
    : theta(t), X(std::move(x)), Y(std::move(y)) {}

  /// Defaulted copy/move assignment operators.
  DeLongPlacementsRes(const DeLongPlacementsRes&) = default;
  DeLongPlacementsRes& operator=(const DeLongPlacementsRes&) = default;
  DeLongPlacementsRes(DeLongPlacementsRes&&) noexcept = default;
  DeLongPlacementsRes& operator=(DeLongPlacementsRes&&) noexcept = default;
};

/**
 * @brief Represents the result of causal pattern analysis between two signature spaces.
 *
 * This structure stores both per-sample and aggregated causality measures derived
 * from pattern-space interactions. It is designed to capture symbolic dynamics
 * relationships between real and predicted signals.
 *
 * Fields include:
 * - **NoCausality, PositiveCausality, NegativeCausality, DarkCausality**:
 *   Per-sample causal strengths categorized by type.
 * - **RealLoop**: Index list of valid samples actually processed.
 * - **PatternTypes**: Encoded causal type for each sample (0=no, 1=pos, 2=neg, 3=dark).
 * - **matrice**: Final averaged causality heatmap.
 * - **TotalPos, TotalNeg, TotalDark**: Global mean strengths for each category.
 */
struct PatternCausalityRes {
  std::vector<double> NoCausality;         ///< Strengths classified as "no causality".
  std::vector<double> PositiveCausality;   ///< Strengths classified as "positive causality".
  std::vector<double> NegativeCausality;   ///< Strengths classified as "negative causality".
  std::vector<double> DarkCausality;       ///< Strengths classified as "dark causality".
  std::vector<size_t> RealLoop;            ///< Indices of valid samples actually processed.
  std::vector<int> PatternTypes;           ///< Encoded pattern types (0–3).
  std::vector<std::string> PatternStrings; ///< Encoded diff strings ("123","312",etc).
  std::vector<std::vector<double>> matrice;///< Averaged heatmap (pattern × pattern).
  double TotalPos = std::numeric_limits<double>::quiet_NaN(); ///< Global mean of diagonal elements.
  double TotalNeg = std::numeric_limits<double>::quiet_NaN(); ///< Global mean of anti-diagonal elements.
  double TotalDark = std::numeric_limits<double>::quiet_NaN();///< Global mean of off-diagonal elements.

  /// Default constructor: initializes empty result with NaN totals.
  PatternCausalityRes() = default;

  /// Copy constructor.
  PatternCausalityRes(const PatternCausalityRes&) = default;

  /// Move constructor (noexcept for efficiency in containers).
  PatternCausalityRes(PatternCausalityRes&&) noexcept = default;

  /// Copy assignment.
  PatternCausalityRes& operator=(const PatternCausalityRes&) = default;

  /// Move assignment.
  PatternCausalityRes& operator=(PatternCausalityRes&&) noexcept = default;
};

#endif // DataStruct_H
