#'@title FFT Filter
#'@description Frequency-domain smoothing using the Fast Fourier Transform
#' (FFT) to attenuate high-frequency components.
#'
#'@return A `ts_fil_fft` object.
#'
#'@details The implementation estimates a cutoff based on spectral statistics
#' and reconstructs the series from dominant frequencies.
#'
#'@references
#' - J. W. Cooley and J. W. Tukey (1965). An algorithm for the machine
#'   calculation of complex Fourier series. Math. Comput.
#'@examples
#'# Frequency-domain smoothing via FFT cutoff
#' # Load package and example data
#' library(daltoolbox)
#' data(tsd)
#' tsd$y[9] <- 2 * tsd$y[9]  # inject an outlier
#'
#' # Fit FFT-based filter and reconstruct without high frequencies
#' filter <- ts_fil_fft()
#' filter <- fit(filter, tsd$y)
#' y <- transform(filter, tsd$y)
#'
#' # Compare original vs frequency-smoothed series
#' plot_ts_pred(y = tsd$y, yadj = y)
#'@importFrom daltoolbox dal_transform
#'@importFrom daltoolbox fit
#'@importFrom daltoolbox transform
#'@export
ts_fil_fft <- function() {
  obj <- dal_transform()
  class(obj) <- append("ts_fil_fft", class(obj))
  return(obj)
}

compute_cut_index <- function(freqs) {
  # Initial cutoff at dominant frequency; adjust by threshold if spectrum varies
  cutindex <- which.max(freqs)
  if (min(freqs) != max(freqs)) {
    threshold <- mean(freqs) + 2.968 * sd(freqs)
    freqs[freqs < threshold] <- min(freqs) + max(freqs)
    cutindex <- which.min(freqs)
  }
  return(cutindex)
}


#'@importFrom daltoolbox transform
#'@importFrom stats fft
#'@importFrom stats sd
#'@exportS3Method transform ts_fil_fft
transform.ts_fil_fft <- function(obj, data, ...) {

  fft_signal <- stats::fft(data)

  spectrum <- base::Mod(fft_signal) ^ 2
  half_spectrum <- spectrum[1:(length(obj$serie) / 2 + 1)]

  cutindex <- compute_cut_index(half_spectrum)
  n <- length(fft_signal)

  fft_signal[1:cutindex] <- 0
  fft_signal[(n - cutindex):n] <- 0

  noise <- base::Re(stats::fft(fft_signal, inverse = TRUE) / n)

  result <- data - noise

  return(result)
}

