% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{fit}
\alias{fit}
\title{Calculate Fit Indices}
\usage{
fit(
  Y,
  Q,
  model = "GDINA",
  att.str = NULL,
  mono.constraint = FALSE,
  maxitr = 2000,
  verbose = 1
)
}
\arguments{
\item{Y}{A required \eqn{N} × \eqn{I} matrix or \code{data.frame} consisting of the responses of \code{N} individuals
to \eqn{I} items. Missing values need to be coded as \code{NA}.}

\item{Q}{A required binary \eqn{I} × \eqn{K} matrix containing the attributes not required or required 
master the items. The \code{i}th row of the matrix is a binary indicator vector indicating which
attributes are not required (coded by 0) and which attributes are required (coded by 1) to master
item \eqn{i}.}

\item{model}{Type of model to be fitted; can be \code{"GDINA"}, \code{"LCDM"}, \code{"DINA"}, \code{"DINO"},
\code{"ACDM"}, \code{"LLM"}, or \code{"rRUM"}. Default = \code{"GDINA"}.}

\item{att.str}{Specify attribute structures. \code{NULL}, by default, means there is no structure. Attribute structure 
needs be specified as a list - which will be internally handled by \code{\link[GDINA]{att.structure}} function. 
See examples. It can also be a matrix giving all permissible attribute profiles.}

\item{mono.constraint}{Logical indicating whether monotonicity constraints should be fulfilled in estimation.
Default = \code{FALSE}.}

\item{maxitr}{A vector for each item or nonzero category, or a scalar which will be used for all items
to specify the maximum number of EM cycles allowed. Default = \code{2000}.}

\item{verbose}{Can be \code{0}, \code{1} or \code{2}, indicating to print no information, information
for current iteration, or information for all iterations. Default = \code{1}.}
}
\value{
An object of class \code{fit}. The \code{fit} contains various fit indices:
\describe{
 \item{\code{npar}}{The number of parameters.}
 \item{\code{-2LL}}{The Deviance.}
 \item{\code{AIC}}{The Akaike information criterion.}
 \item{\code{BIC}}{The Bayesian information criterion.}
 \item{\code{CAIC}}{The consistent Akaike information criterion.}
 \item{\code{SABIC}}{The Sample-size Adjusted BIC.}
 \item{\code{M2}}{A vector consisting of \eqn{M_2} statistic, degrees of freedom, significance level, and \eqn{RMSEA_2} (Liu, Tian, & Xin, 2016).}
 \item{\code{SRMSR}}{The standardized root mean squared residual (SRMSR; Ravand & Robitzsch, 2018).}
 \item{\code{arguments}}{A list containing all input arguments}
}
}
\description{
Calculate relative fit indices (-2LL, AIC, BIC, CAIC, SABIC) and absolute fit indices (\eqn{M_2} test, \eqn{RMSEA_2}, SRMSR) 
using the \code{\link[GDINA]{modelfit}} function in the \code{GDINA} package.
}
\examples{
\donttest{
set.seed(123)

library(Qval)

## generate Q-matrix and data to fit
K <- 5
I <- 30
Q <- sim.Q(K, I)
IQ <- list(
  P0 = runif(I, 0.0, 0.2),
  P1 = runif(I, 0.8, 1.0)
)
data <- sim.data(Q = Q, N = 500, IQ = IQ, model = "GDINA", distribute = "horder")

## calculate fit indices
fit.indices <- fit(Y = data$dat, Q = Q, model = "GDINA")
print(fit.indices)
}

}
\references{
Khaldi, R., Chiheb, R., & Afa, A.E. (2018). Feed-forward and Recurrent Neural Networks for Time Series Forecasting: Comparative Study. In: Proceedings of the International Conference on Learning and Optimization Algorithms: Theory and Applications (LOPAL 18). Association for Computing Machinery, New York, NY, USA, Article 18, 1–6. DOI: 10.1145/3230905.3230946.

Liu, Y., Tian, W., & Xin, T. (2016). An application of M2 statistic to evaluate the fit of cognitive diagnostic models. Journal of Educational and Behavioral Statistics, 41, 3–26. DOI: 10.3102/1076998615621293.

Ravand, H., & Robitzsch, A. (2018). Cognitive diagnostic model of best choice: a study of reading comprehension. Educational Psychology, 38, 1255–1277. DOI: 10.1080/01443410.2018.1489524.
}
\author{
Haijiang Qin <Haijiang133@outlook.com>
}
