% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyze_and_visualize_uhi.R
\name{analyze_and_visualize_uhi}
\alias{analyze_and_visualize_uhi}
\title{Urban Heat Island Analysis & Visualization}
\usage{
analyze_and_visualize_uhi(
  location,
  date_range = c("2023-06-01", "2023-08-31"),
  hex_resolution = 9,
  ghsl_path = NULL,
  tree_canopy_radius = 5,
  thermal_source = c("auto", "ecostress", "landsat"),
  composite_scenes = FALSE,
  max_scenes = 5,
  lst_percentile_filter = c(0.01, 0.99),
  correlation_method = c("spearman", "pearson"),
  use_exactextract = TRUE,
  parallel = FALSE,
  n_cores = NULL
)
}
\arguments{
\item{location}{Character or numeric vector. Either a city name (e.g., "Paris, France")
or a bounding box as c(xmin, ymin, xmax, ymax) in EPSG:4326.}

\item{date_range}{Character vector of length 2. Date range for satellite imagery
in ISO format, e.g., c("2023-06-01", "2023-08-31"). Summer months recommended
for UHI analysis.}

\item{hex_resolution}{Integer. H3 hexagon resolution (default 9).
Higher values = smaller hexagons. Range: 0-15.}

\item{ghsl_path}{Character or NULL. Path to GHSL Built-S raster (.tif) for built-up
coverage. If NULL (default), uses OSM buildings as fallback.}

\item{tree_canopy_radius}{Numeric. Buffer radius for tree points in meters (default 5).
Represents approximate canopy spread.}

\item{thermal_source}{Character. One of "auto", "ecostress", or "landsat".
Default "auto" tries ECOSTRESS first, then Landsat.}

\item{composite_scenes}{Logical. Whether to composite multiple satellite scenes
using median (default FALSE). Useful for reducing cloud gaps.}

\item{max_scenes}{Integer. Maximum number of scenes to composite if composite_scenes=TRUE
(default 5).}

\item{lst_percentile_filter}{Numeric vector of length 2 or NULL. Lower and upper
percentiles for LST outlier removal (default c(0.01, 0.99)).
Set to NULL to disable filtering.}

\item{correlation_method}{Character. Correlation method: "spearman" (default, robust)
or "pearson".}

\item{use_exactextract}{Logical. Use exactextractr package for faster raster extraction
if available (default TRUE). Falls back to terra::extract if not installed.}

\item{parallel}{Logical. Reserved for future parallel processing (currently ignored).}

\item{n_cores}{Integer or NULL. Reserved for future use (currently ignored).}
}
\value{
A list with the following components:
\describe{
\item{results}{An sf object with hexagon-level results including LST_mean, LST_diff,
Green_Pct, Built_Pct, Water_Pct, Gi_Star, Hotspot_Category, etc.}
\item{maps}{A list of map objects:
\itemize{
\item interactive: Leaflet map with all layers
\item lst, deviation, green, built, hotspot: Individual ggplot2 maps
\item combined: Patchwork combined static map
\item scatter: Scatter plot dashboard
}
}
\item{stats}{A list with descriptive statistics, correlations, regression results,
and spatial autocorrelation (Moran's I)}
\item{meta}{Metadata including location, data sources, processing parameters,
and timing information}
\item{export_geojson}{Function to export results to GeoJSON}
\item{export_results}{Function to export results to multiple formats (geojson, gpkg, csv, shp)}
}
}
\description{
Performs a comprehensive UHI analysis: fetches thermal data (ECOSTRESS or Landsat),
retrieves environmental data using \code{get_osm_data} (green spaces, trees, highways),
computes green coverage using fast raster-based methods, calculates built-up coverage,
performs spatial hotspot analysis, and generates interactive and static maps.
}
\details{
\strong{CRAN policy note:} This function downloads data from the internet (OpenStreetMap,
Microsoft Planetary Computer STAC API for satellite imagery). Internet access is
required for this function to work. The function will fail gracefully with
informative error messages if network access is unavailable.

\strong{Data Sources:}
\itemize{
\item Land Surface Temperature: ECOSTRESS (70m) or Landsat 8/9 (100m thermal) via
Microsoft Planetary Computer STAC API
\item Green spaces, trees, buildings, water bodies: OpenStreetMap via osmdata
\item Optional: GHSL Built-S raster for built-up coverage
\item Optional: NDVI from Landsat for satellite-based vegetation index
}

\strong{Analysis Components:}
\itemize{
\item H3 hexagonal grid aggregation at configurable resolution
\item Green coverage from OSM polygons, tree points (with canopy buffer), and NDVI
\item Built-up coverage from GHSL or OSM buildings (fallback)
\item Water body masking to exclude water hexagons from land-based analyses
\item Getis-Ord Gi* hotspot analysis with significance testing
\item Moran's I spatial autocorrelation
\item Correlation and regression analysis (LST vs Green/Built)
}

\strong{Output Maps:}
\itemize{
\item Interactive Leaflet map with toggleable layers (LST, Deviation, Green, Built, Hotspots)
\item Static ggplot2 maps for publication
\item Scatter plot dashboard with regression diagnostics
}
}
\examples{
\dontrun{
# Basic usage with city name
result <- analyze_and_visualize_uhi(
  location = "Basel, Switzerland",
  date_range = c("2023-06-01", "2023-08-31")
)

# View interactive map
result$maps$interactive

# View static combined map
print(result$maps$combined)

# View statistics
print(result$stats$descriptive)
print(result$stats$correlations)

# Export results
result$export_geojson("basel_uhi_results.geojson")
result$export_results("basel_uhi", formats = c("geojson", "csv"))

# Using bounding box instead of city name
result_bbox <- analyze_and_visualize_uhi(
  location = c(7.55, 47.53, 7.65, 47.58),  # Basel area
  date_range = c("2023-07-01", "2023-07-31"),
  thermal_source = "landsat",
  hex_resolution = 9  # Larger hexagons
)

# With GHSL built-up data for more accurate built coverage
result_ghsl <- analyze_and_visualize_uhi(
  location = "Zurich, Switzerland",
  date_range = c("2023-06-01", "2023-08-31"),
  ghsl_path = "path/to/ghsl_built.tif"
)
}

}
