% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/separate.R
\name{gr_separate}
\alias{gr_separate}
\title{Advanced hydrograph separation}
\usage{
gr_separate(df, params = gr_get_params(), debug = FALSE)
}
\arguments{
\item{df}{\code{data.frame} with four columns: date, runoff, temperature, precipitation.}

\item{params}{\code{list} of separation parameters, as returned by \code{\link[=gr_get_params]{gr_get_params()}} function. Can also be a \code{list} of such \code{list}s if modified parameters are required for some years. In this case the length of \code{params} must be equal to the number of calendar years in \code{df} or be equal to \code{1}.}

\item{debug}{Boolean. If \code{TRUE} then additional attributes \code{jittered} and \code{params} are written to the output \code{data.frame}. \code{jittered} is an integer vector of years for which the separation parameters were randomly jittered. \code{params} is a list of separation parameter lists used for each year (some o those may have been jittered). Defaults to \code{FALSE}.}
}
\value{
A \code{data.frame} with 11 columns:\tabular{ll}{
   \strong{Column} \tab \strong{Description} \cr
   \code{Date} \tab date \cr
   \code{Q} \tab total runoff \cr
   \code{Temp} \tab temperature \cr
   \code{Prec} \tab precipitation \cr
   \code{Qbase} \tab baseflow \cr
   \code{Quick} \tab quickflow \cr
   \code{Qspri} \tab spring flood \cr
   \code{Qrain} \tab rain floods \cr
   \code{Qthaw} \tab thaw floods \cr
   \code{Season} \tab a season of the year \cr
   \code{Year} \tab a water-resources year \cr
}
}
\description{
Separates the runoff into genetic components: groundwater, thaw, rain and spring.
}
\examples{
library(grwat)

data(spas) # example Spas-Zagorye data is included with grwat package
head(spas)

# separate
sep = gr_separate(spas, params = gr_get_params(reg = 'center'))

# Visualize
gr_plot_sep(sep, c(1978, 1989)) 

# Debug mode gives access to additional information
sep_debug = gr_separate(spas, 
                        params = gr_get_params(reg = 'center'), 
                        debug = TRUE)

# a vector of years with jittered params
jit = attributes(sep_debug)$jittered
print(jit)

# actual params used for each year
parlist = attributes(sep_debug)$params

# tabular representation of parameters
partab = gr_to_pardf(parlist)
head(partab)

parlist2 = partab |> 
  dplyr::select(-year) |> 
  apply(1, as.list) |> 
  lapply(\(X) {
    n = length(X)
    X[1:(n - 1)] <- lapply(X[1:(n - 1)], as.numeric)
    return(X)
  }) |> 
  setNames(partab$year)

# extract and tweak parameters for selected year
p = parlist[['1989']]
p$grad1 = 1
p$grad2 = 2.5

# use tweaked parameters for all years
sep_debug = gr_separate(spas, params = p, debug = TRUE)

# Visualize
gr_plot_sep(sep_debug, c(1978, 1989)) 

# actual params used for each year
parlist = attributes(sep_debug)$params

# tweak parameters for selected year
parlist[['1989']]$grad1 = 3
parlist[['1989']]$grad2 = 6

# set the sprecdays parameter for multiple years
parlist = gr_set_param(parlist, sprecdays, 
                       years = c(1978, 1989:1995), 
                       value = 15)

# set the spcomp parameter for all years
parlist = gr_set_param(parlist, spcomp, value = 2.5)

# use the list of parameters for separation
sep_debug = gr_separate(spas, params = parlist, debug = TRUE)

# Visualize
gr_plot_sep(sep_debug, c(1978, 1989))



}
