/*
# MIT License
#
# Copyright (c) 2025 Reed A. Cartwright <racartwright@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
*/

/*
# Documentation

Ironseed is a finite-entropy (or fixed-entropy) hash that can be used to
generate an unlimited sequence of seeds for initializing the state of a random
number generator. It is inspired by the work of M.E. O'Neill and others [1,2].

An ironseed is a 256-bit hash constructed from a variable-length sequence of
32-bit inputs. Each ironseed consists of eight 32-bit sub-hashes. The sub-hashes
are 32-bit multilinear hashes [3] that accumulate entropy from the input
sequence. Each input is included in every sub-hash. The coefficients for the
multilinear hashes are generated by a Weyl sequence.

Multilinear hashes are also used to generate a seed sequence from an ironseed.
Each 32-bit output value is generated by uniquely hashing the sub-hashes. The
coefficients for the output are also generated by a second Weyl sequence.

To improve the observed randomness of each hash output, bits are mixed using
a finalizer adapted from SplitMix64 [4]. With the additional mixing from the
finalizer, the output seed sequence passes PractRand tests [5].

[1] https://www.pcg-random.org/posts/developing-a-seed_seq-alternative.html
[2] https://gist.github.com/imneme/540829265469e673d045
[3] https://arxiv.org/pdf/1202.4961.pdf
[4] https://doi.org/10.1145/2714064.2660195
[5] https://pracrand.sourceforge.net/
*/
#define R_NO_REMAP

#include <R.h>
#include <R_ext/Visibility.h>
#include <Rinternals.h>
#include <assert.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

typedef struct ironseed_hash {
  uint64_t coef;
  uint64_t hashes[8];
} ironseed_hash_t;

typedef struct ironseed {
  uint32_t seed[8];
} ironseed_t;

/* The following constants were selected by generating four random primes in
 * Mathematica:
 *
 *     IntegerString[NextPrime[RandomInteger[{0, 2^64-1}]], 16]
 *
 * Constants A and C are generated primes, and constants B and D are the
 * generated primes multiplied by two.
 *
 * These constants are used to setup two Weyl sequences that are always odd and
 * importantly never zero.
 */
static const uint64_t IRONSEED_A = 0xc9f736a1a00d1f5f;
static const uint64_t IRONSEED_B = 0x1044d9bc1bd04d7e;
static const uint64_t IRONSEED_C = 0x278abb429678dd43;
static const uint64_t IRONSEED_D = 0xf55176215fdee4b6;

// Multilinear hash (https://arxiv.org/pdf/1202.4961.pdf)
// Hash is based on a sequence of 64-bit numbers generated by Weyl sequence
// Multilinear hash is (m_0 + sum(m_i*u_i) mod 2^64) / 2^32
// m = buffer of 64-bit unsigned random values
// u = 32-bit input values that are being hashed

static inline uint64_t init_hash4i_coef(void) { return IRONSEED_A; }
static inline uint64_t init_hash4o_coef(void) { return IRONSEED_C; }

static inline uint64_t hash4i_coef(uint64_t *m) {
  uint64_t r = *m;
  *m += IRONSEED_B;
  return r;
}

static inline uint64_t hash4o_coef(uint64_t *m) {
  uint64_t r = *m;
  *m += IRONSEED_D;
  return r;
}

// Adapted from mix32 of Java's SplittableRandom algorithm
// This is variant 4 of Stafford's mixing algorithms.
// http://zimbry.blogspot.com/2011/09/better-bit-mixing-improving-on.html
static inline uint32_t finalmix(uint64_t u) {
  u = (u ^ (u >> 33)) * 0x62a9d9ed799705f5;
  u = (u ^ (u >> 28)) * 0xcb24d0a5c88c35b3;
  return u >> 32;
}

// Initialize an ironseed_hash object. Stores the intermediate values of
// 8 multilinear hashes as additional values are added to the hash.
static void init_ironseed_hash(ironseed_hash_t *p) {
  assert(p != NULL);
  p->coef = init_hash4i_coef();
  for(int i = 0; i < 8; ++i) {
    p->hashes[i] = hash4i_coef(&p->coef);
  }
}

// support for big-endian systems
#define byteswap32(x) __builtin_bswap32(x)

#ifndef WORDS_BIGENDIAN
#define htol32(x) (x)
#else
#define htol32(x) byteswap32(x)
#endif

// Update the intermediate state of each sub-hash.
static void update_ironseed_hash(ironseed_hash_t *p, uint32_t value) {
  assert(p != NULL);
  for(int i = 0; i < 8; ++i) {
    p->hashes[i] += hash4i_coef(&p->coef) * value;
  }
}

static void update_ironseed_hash_ll(ironseed_hash_t *p, uint64_t value) {
  assert(p != NULL);
  update_ironseed_hash(p, (uint32_t)value);
  update_ironseed_hash(p, (uint32_t)(value >> 32));
}

static void update_ironseed_hash_d(ironseed_hash_t *p, double value) {
  assert(p != NULL);
  uint64_t u;
  memcpy(&u, &value, sizeof(u));
  update_ironseed_hash_ll(p, u);
}

static void update_ironseed_hash_p(ironseed_hash_t *p, void *value) {
  update_ironseed_hash_ll(p, (uint64_t)((uintptr_t)value));
}

static void update_ironseed_hash_f(ironseed_hash_t *p, DL_FUNC value) {
  update_ironseed_hash_ll(p, (uint64_t)((uintptr_t)value));
}

static void update_ironseed_hash_v(
  ironseed_hash_t *p, const void *buffer, size_t len
) {
  assert(p != NULL);
  assert(buffer != NULL);

  size_t i = 0;

  // Hash lower 32-bits of length since it may not be a multiple of 4.
  update_ironseed_hash(p, (uint32_t)len);

  // Exit early if length is 0.
  if(len == 0) {
    return;
  }

  // Turn buffer into an array of integers using little-endian format.
  const char *b = buffer;
  for(; i + 4 < len; i += 4) {
    uint32_t u;
    memcpy(&u, b + i, 4);
    update_ironseed_hash(p, htol32(u));
  }
  uint32_t u = 0;
  memcpy(&u, b + i, len - i);
  update_ironseed_hash(p, htol32(u));
}

static void update_ironseed_hash_s(ironseed_hash_t *p, const char *s) {
  if(s == NULL) {
    return;
  }
  update_ironseed_hash_v(p, s, strlen(s));
}

static void create_ironseed(const ironseed_hash_t *p, ironseed_t *v) {
  assert(p != NULL);
  assert(v != NULL);

  // If the last valued hashed was 0, then it did not affect the hash
  // value. It did affect the coefficients. Append one extra value to
  // the end before we generate the result of each hash.
  uint64_t k = p->coef;
  for(int i = 0; i < 8; ++i) {
    uint64_t u = p->hashes[i] + hash4i_coef(&k);
    v->seed[i] = finalmix(u);
  }
}

// defined in compat.c
uint64_t clock_entropy(void);
uint64_t pid_entropy(void);
uint64_t tid_entropy(void);
uint64_t readcycle_entropy(void);
uint64_t system_entropy(void);
void hostname_entropy(char *name, size_t size);

// Inspired by ideas from M.E. O'Neill
// - https://www.pcg-random.org/posts/simple-portable-cpp-seed-entropy.html
// - https://gist.github.com/imneme/540829265469e673d045
//
// TODO: extract entropy from cluster environmental variables like JOB_ID
static void autofill_ironseed_hash(ironseed_hash_t *p) {
  assert(p != NULL);

  char buffer[256] = "";

  // string that changes every time this file is compiled
  const char compile_stamp[] = __DATE__ __TIME__ __FILE__;
  update_ironseed_hash_s(p, compile_stamp);

  // heap and stack randomness
  void *malloc_addr = malloc(sizeof(int));
  void *stack_addr = &malloc_addr;
  update_ironseed_hash_p(p, malloc_addr);
  update_ironseed_hash_p(p, stack_addr);
  free(malloc_addr);

  // addresses of a few functions
  update_ironseed_hash_f(p, (DL_FUNC)&clock);
  update_ironseed_hash_f(p, (DL_FUNC)&getenv);
  update_ironseed_hash_f(p, (DL_FUNC)&autofill_ironseed_hash);
  update_ironseed_hash_f(p, (DL_FUNC)&Rf_allocVector);

  // time randomness
  update_ironseed_hash_ll(p, clock_entropy());

  // process id
  update_ironseed_hash_ll(p, pid_entropy());

  // thread id
  update_ironseed_hash_ll(p, tid_entropy());

  // readcyclecounter
  update_ironseed_hash_ll(p, readcycle_entropy());

  // os entropy
  update_ironseed_hash_ll(p, system_entropy());

  // hostname entropy
  hostname_entropy(buffer, sizeof(buffer));
  update_ironseed_hash_s(p, buffer);

  // Job ID entropy on clusters
  const char *s;
  if((s = getenv("SLURM_JOB_ID")) != NULL) {
    update_ironseed_hash_s(p, s);
    update_ironseed_hash_s(p, getenv("SLURM_JOB_NAME"));
    update_ironseed_hash_s(p, getenv("SLURM_ARRAY_TASK_ID"));
  } else if((s = getenv("PBS_JOBID")) != NULL) {
    update_ironseed_hash_s(p, s);
    update_ironseed_hash_s(p, getenv("PBS_JOBNAME"));
  } else if((s = getenv("LSB_JOBID")) != NULL) {
    update_ironseed_hash_s(p, s);
    update_ironseed_hash_s(p, getenv("LSB_JOBNAME"));
    update_ironseed_hash_s(p, getenv("LSB_JOBINDEX"));
  } else if((s = getenv("FLUX_JOB_ID")) != NULL) {
    update_ironseed_hash_s(p, s);
  } else if((s = getenv("JOB_ID")) != NULL) {
    update_ironseed_hash_s(p, s);
    update_ironseed_hash_s(p, getenv("JOB_NAME"));
    update_ironseed_hash_s(p, getenv("SGE_TASK_ID"));
  } else if((s = getenv("AWS_BATCH_JOB_ID")) != NULL) {
    update_ironseed_hash_s(p, s);
    update_ironseed_hash_s(p, getenv("AWS_BATCH_JOB_ATTEMPT"));
    update_ironseed_hash_s(p, getenv("AWS_BATCH_JOB_ARRAY_INDEX"));
  }
}

static void create_seedseq(
  const ironseed_t *p, uint64_t *m, unsigned int *u, size_t len
) {
  assert(p != NULL);
  assert(u != NULL);
  assert(m != NULL);

  for(size_t i = 0; i < len; ++i) {
    uint64_t v = hash4o_coef(m);
    for(int j = 0; j < 8; ++j) {
      v += hash4o_coef(m) * p->seed[j];
    }
    u[i] = finalmix(v);
  }
}

SEXP R_create_ironseed(SEXP x) {
  // initialize hash
  ironseed_hash_t hash;
  init_ironseed_hash(&hash);

  // has information provided by x
  for(R_xlen_t i = 0; i < XLENGTH(x); ++i) {
    SEXP y = VECTOR_ELT(x, i);
    switch(TYPEOF(y)) {
    case INTSXP:
      for(R_xlen_t j = 0; j < XLENGTH(y); ++j) {
        update_ironseed_hash(&hash, INTEGER(y)[j]);
      }
      break;
    case REALSXP:
      for(R_xlen_t j = 0; j < XLENGTH(y); ++j) {
        update_ironseed_hash_d(&hash, REAL(y)[j]);
      }
      break;
    case CPLXSXP:
      for(R_xlen_t j = 0; j < XLENGTH(y); ++j) {
        update_ironseed_hash_d(&hash, COMPLEX(y)[j].r);
        update_ironseed_hash_d(&hash, COMPLEX(y)[j].i);
      }
      break;
    case LGLSXP:
      for(R_xlen_t j = 0; j < XLENGTH(y); ++j) {
        update_ironseed_hash(&hash, LOGICAL(y)[j]);
      }
      break;
    case STRSXP:
      for(R_xlen_t j = 0; j < XLENGTH(y); ++j) {
        update_ironseed_hash_s(&hash, Rf_translateCharUTF8(STRING_ELT(y, j)));
      }
      break;
    case RAWSXP:
      update_ironseed_hash_v(&hash, RAW(y), XLENGTH(y));
      break;
    case NILSXP:
      break;
    default:
      Rf_error(
        "Ironseed hash: unsupported type `%s`",
        CHAR(Rf_type2str_nowarn(TYPEOF(y)))
      );
    }
  }

  // Construct ironseed
  ironseed_t seed;
  create_ironseed(&hash, &seed);

  SEXP ret = PROTECT(Rf_allocVector(INTSXP, 8));
  for(int i = 0; i < 8; ++i) {
    INTEGER(ret)[i] = seed.seed[i];
  }

  Rf_classgets(ret, Rf_ScalarString(Rf_mkCharCE("ironseed_ironseed", CE_UTF8)));
  UNPROTECT(1);
  return ret;
}

SEXP R_auto_ironseed(void) {
  ironseed_hash_t hash;
  init_ironseed_hash(&hash);
  autofill_ironseed_hash(&hash);

  ironseed_t seed;
  create_ironseed(&hash, &seed);

  SEXP ret = PROTECT(Rf_allocVector(INTSXP, 8));
  for(int i = 0; i < 8; ++i) {
    INTEGER(ret)[i] = seed.seed[i];
  }

  Rf_classgets(ret, Rf_ScalarString(Rf_mkCharCE("ironseed_ironseed", CE_UTF8)));
  UNPROTECT(1);
  return ret;
}

SEXP R_create_seedseq(SEXP x, SEXP n, SEXP k) {
  union {
    uint64_t u;
    double d;
  } u = {init_hash4o_coef()};

  int len = Rf_asInteger(n);

  if(!Rf_isNull(k)) {
    u.d = Rf_asReal(k);
  }

  ironseed_t seed;
  for(int i = 0; i < 8; ++i) {
    seed.seed[i] = INTEGER(x)[i];
  }

  SEXP ret = PROTECT(Rf_allocVector(INTSXP, len));

  create_seedseq(&seed, &u.u, (unsigned int *)INTEGER(ret), XLENGTH(ret));

  Rf_setAttrib(ret, Rf_install("k"), Rf_ScalarReal(u.d));

  UNPROTECT(1);
  return ret;
}
