% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bland_altman.R
\name{bland_altman}
\alias{bland_altman}
\alias{print.ba}
\alias{plot.ba}
\title{Bland-Altman statistics with confidence intervals}
\usage{
bland_altman(
  group1,
  group2,
  two = 1.96,
  mode = 1L,
  conf_level = 0.95,
  verbose = FALSE
)

\method{print}{ba}(x, digits = 3, ci_digits = 3, ...)

\method{plot}{ba}(
  x,
  title = "Bland-Altman Plot",
  subtitle = NULL,
  point_alpha = 0.7,
  point_size = 2.2,
  line_size = 0.8,
  shade_ci = TRUE,
  shade_alpha = 0.08,
  smoother = c("none", "loess", "lm"),
  symmetrize_y = TRUE,
  ...
)
}
\arguments{
\item{group1, group2}{Numeric vectors of equal length.}

\item{two}{Positive scalar; the multiple of the standard deviation used to
define the LoA (default 1.96 for nominal 95\\% agreement). The confidence
intervals always use \eqn{t_{n-1,\,1-\alpha/2}} regardless of this choice.}

\item{mode}{Integer; 1 uses \code{group1 - group2}, 2 uses \code{group2 - group1}.}

\item{conf_level}{Confidence level for CIs (default 0.95).}

\item{verbose}{Logical; if TRUE, prints how many OpenMP threads are used.}

\item{x}{A \code{"ba"} object.}

\item{digits}{Number of digits for estimates (default 3).}

\item{ci_digits}{Number of digits for CI bounds (default 3).}

\item{...}{Passed to \code{ggplot2::theme()} (ggplot path) or \code{plot()}.}

\item{title}{Plot title.}

\item{subtitle}{Optional subtitle. If NULL, shows n and LoA summary.}

\item{point_alpha}{Point transparency.}

\item{point_size}{Point size.}

\item{line_size}{Line width for mean/LoA.}

\item{shade_ci}{Logical; if TRUE, draw shaded CI bands instead of 6 dashed
lines.}

\item{shade_alpha}{Transparency of CI bands.}

\item{smoother}{One of "none", "loess", "lm" to visualize proportional bias.}

\item{symmetrize_y}{Logical; if TRUE, y-axis centered at mean difference
with symmetric limits.}
}
\value{
An object of class \code{"ba"} (list) with elements:
\itemize{
\item \code{means}, \code{diffs}: numeric vectors
\item \code{groups}: data.frame used after NA removal
\item \code{based.on}: integer, number of pairs used
\item \code{lower.limit}, \code{mean.diffs}, \code{upper.limit}
\item \code{lines}: named numeric vector (lower, mean, upper)
\item \code{CI.lines}: named numeric vector for CIs of those lines
\item \code{two}, \code{critical.diff}
}
}
\description{
Computes Bland-Altman mean difference and limits of agreement (LoA)
between two numeric measurement vectors, including t-based confidence
intervals for the mean difference and for each LoA using 'C++' backend.

Note: Lin's concordance correlation coefficient (CCC) is a complementary,
single-number summary of agreement (precision + accuracy). It is useful for
quick screening or reporting an overall CI, but may miss systematic or
magnitude-dependent bias; consider reporting CCC alongside Bland-Altman.
}
\details{
Given paired measurements \eqn{(x_i, y_i)}, Bland-Altman analysis uses
\eqn{d_i = x_i - y_i} (or \eqn{y_i - x_i} if \code{mode = 2}) and
\eqn{m_i = (x_i + y_i)/2}. The mean difference \eqn{\bar d} estimates bias.
The limits of agreement (LoA) are \eqn{\bar d \pm z \cdot s_d}, where
\eqn{s_d} is the sample standard deviation of \eqn{d_i} and \eqn{z}
(argument \code{two}) is typically 1.96 for nominal 95\% LoA.

Confidence intervals use Student's \eqn{t} distribution with \eqn{n-1}
degrees of freedom, with
\itemize{
\item Mean-difference CI given by \eqn{\bar d \pm t_{n-1,\,1-\alpha/2}\,
  s_d/\sqrt{n}}; and
\item LoA CI given by \eqn{(\bar d \pm z\, s_d) \;\pm\;
  t_{n-1,\,1-\alpha/2}\, s_d\,\sqrt{3/n}}.
}

Assumptions include approximately normal differences and roughly constant
variability across the measurement range; if differences increase with
magnitude, consider a transformation before analysis. Missing values are
removed pairwise (rows with an \code{NA} in either input are dropped before
calling the C++ backend).
}
\examples{
set.seed(1)
x <- rnorm(100, 100, 10)
y <- x + rnorm(100, 0, 8)
ba <- bland_altman(x, y)
print(ba)
plot(ba)

}
\references{
Bland JM, Altman DG (1986). Statistical methods for assessing agreement
between two methods of clinical measurement. \emph{The Lancet}, 307-310.

Bland JM, Altman DG (1999). Measuring agreement in method comparison studies.
\emph{Statistical Methods in Medical Research}, 8(2), 135-160.
}
\seealso{
\code{\link{print.ba}}, \code{\link{plot.ba}},
\code{\link{ccc}},\code{\link{ccc_pairwise_u_stat}},
\code{\link{ccc_lmm_reml}}
}
\author{
Thiago de Paula Oliveira
}
