#' Stitch images
#'
#' \code{stitchImgs} - This function stitches all specified images (currently
#' only PNGs supported) together in a line.
#' It also searches for the best overlap between two consecutive
#' images (see parameters and  \code{findOverlap()}).
#' If \code{overlap_px} is already known, this
#' function calls \code{blendImgs()}.
#'
#' @param img_paths (Optional, default = NULL) Character vector specifying all
#' of the individual image paths of interest. This is only used if
#' \code{imgs} is set to NULL. For \code{ImageCorr()} it must have length 2.
#' @param imgs List of images (e.g., provided by the RootDetector). Each image
#' can be a PNG, i.e., an array with 3 dimensions (3 layers each containing a
#' 2-dim. numeric matrix with values between 0 and 1), or a 2-dim. matrix.
#' For \code{ImageCorr()} it must have length 2.
#' @param out_file Full path for how the stitched image should be
#' saved, e.g. 'C:/path/stitched.png'. If no path is provided, the image is
#' not saved (only returned).
#' @param overlap_px Numeric vector (default NULL) specifying the (likely)
#' widths of the overlaps between two consecutive images. The vector must have
#' one element less than there are images and must not contain any negative
#' values.
#' @param max_shift_px Numeric vector (default NULL) specifying the maximal
#' deviation in pixels from the \code{overlap_px}, i.e., all possible
#' overlaps in that range from the likely overlap are compared and the best
#' is chosen if it is better than the \code{overlap_px} (see also
#' \code{perc_better_per_px}). If \code{overlap_px} is already exact, then set
#' \code{max_shift_px} to zero(s). If at NULL, it is set to 10 percent of the
#' image.
#' @param perc_better_per_px Numeric value (percentage, default 0.01, i.e., 1
#' percent) specifying how much better the correlation of an overlap must be
#' than the \code{overlap_px} per pixel difference, to be selected instead.
#' If set to 0, the overall best correlation determines the overlap.
#' Example: If set to 0.01 = 1 percent, an overlap being 4 pixels away from
#' the specified \code{overlap_px} must have a correlation 4*1 percent better
#' than \code{overlap_px} to be chosen as the better overlap.
#' @param corr_formula Character value specifying the formula to be used
#' (by default 1) for calculating how good an overlap of two images is, i.e.,
#' how similar the two overlapping images are. Available are the following
#' formulas: \cr
#' - "frac_matches_rgb_intensities": Fraction of matching intensities over all
#' three color channels. Only suitable for images with few unique colors.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "frac_matches_colors": Fraction of matching colors in the images. Only
#' suitable for images with few unique colors.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "weighted_matches_b_w": Counts the matches of black and white pixels and
#' weighs them anti-proportional to the black and white pixel frequencies.
#' Both black and white make up half of the correlation score.
#' Only suitable for images with mostly pure black and white pixels.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "weighted_matches_colors": Counts the matches per unique color and weighs
#' them anti-proportional to the frequencies of the colors.
#' Each unique color makes up the same fraction of the correlation score.
#' Only suitable for images with few unique colors.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "1-rel_sqrd_diff_rgb_intensities": One minus the relative squared
#' difference of intensities across all color channels.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "1-rel_abs_diff_rgb_intensities": One minus the relative absolute
#' difference of intensities across all color channels.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "1-rel_eucl_diff_colors" (default): One minus the relative Euclidean
#' differences of colors.
#' Ranges from 0 (no matches) to 1 (full match).
#' @param stitch_direction Character specifying in which order the images
#' should be stitched. Available are: 'left_to_right' (default),
#' 'right_to_left', 'top_to_bottom', and 'bottom_to_top'.
#' @param blending_mode Character value specifying how overlapping pixels are
#' combined. Available are: \cr
#' - "under" (default): The first image(s) dominate(s), and only
#' non-overlapping parts of further images contribute. \cr
#' - "over": The next image completely replaces the previous image in the
#' overlap. \cr
#' - "average": The RGB values of overlapping regions are averaged. \cr
#' - "max": The maximal RGB values from both images are chosen. \cr
#' - "min": The minimal RGB values from both images are chosen.
#' @param show_messages Specify if messages should be depicted (default TRUE).
#'
#' @return \code{stitchImgs} The stitched image in the form of a PNG, i.e.,
#' an array with 3 dimensions (3 layers each containing a 2-dim. numeric matrix
#' with values between 0 and 1).
#' @export
#' @rdname stitchImgs
#'
#' @examples
#' # Small example of stitching two 3x4 matrices left to right together.
#' test <- stitchImgs(imgs = list(matrix(c(1,0,0,0,
#'                                         0,1,0,0,
#'                                         0,0,1,1), ncol = 4, nrow = 3,
#'                                       byrow = TRUE),
#'                                matrix(c(0,0,0,0,
#'                                         1,0,0,1,
#'                                         0,1,1,0), ncol = 4, nrow = 3,
#'                                       byrow = TRUE)),
#'            overlap_px = 1, max_shift_px = 2)
#' # The stitched image also contains the overlaps used for stitching.
#' attributes(test)
stitchImgs <- function(img_paths = NULL, imgs = NULL, out_file = NULL,
                       overlap_px = NULL, max_shift_px = NULL,
                       perc_better_per_px = 0.01,
                       corr_formula = "1-rel_eucl_diff_colors",
                       stitch_direction = "left_to_right",
                       blending_mode = "under",
                       show_messages = TRUE){
  # Find overlap if not already exact. -----------------------------------------
  if(is.null(max_shift_px) || sum(max_shift_px != 0)>0){
    overlap_px <- findOverlap(img_paths = img_paths, imgs = imgs,
                              overlap_px = overlap_px,
                              max_shift_px = max_shift_px,
                              perc_better_per_px = perc_better_per_px,
                              corr_formula = corr_formula,
                              stitch_direction = stitch_direction,
                              show_messages = show_messages)
  }
  # Stitch the images with the (computed) overlaps. ----------------------------
  stitched_img <- blendImgs(img_paths = img_paths, imgs = imgs,
                            out_file = out_file,
                            overlap_px = overlap_px,
                            stitch_direction = stitch_direction,
                            blending_mode = blending_mode)
  attributes(stitched_img)
  return(stitched_img)
}
#' Stitch images
#'
#' \code{blendImgs} - This function stitches all specified images
#' together in a line according to specified overlap widths.
#'
#' @return \code{blendImgs} The stitched image in the form of a
#' PNG, i.e., an array with 3 dimensions (3 layers each containing a 2-dim.
#' numeric matrix with values between 0 and 1).
#' @export
#' @rdname stitchImgs
#'
#' @examples
#' # Example of stitching the two matrices with a fixed (in this case not
#' # optimal) overlap.
#' blendImgs(imgs = list(matrix(c(1,0,0,0,
#'                                0,1,0,0,
#'                                0,0,1,1), ncol = 4, nrow = 3, byrow = TRUE),
#'                       matrix(c(0,0,0,0,
#'                                1,0,0,1,
#'                                0,1,1,0), ncol = 4, nrow = 3, byrow = TRUE)),
#'           overlap_px = 1, blending_mode = "over")[,,1]
blendImgs <- function(img_paths = NULL, imgs = NULL, out_file = NULL,
                      overlap_px, stitch_direction = "left_to_right",
                      blending_mode = "under") {
  # Check input PART I. --------------------------------------------------------
  if(!blending_mode %in% c("under", "over", "average", "max", "min")){
    stop("Unknown 'blending_mode'.")
  }
  if(!stitch_direction %in% c("bottom_to_top", "top_to_bottom",
                              "right_to_left", "left_to_right")){
    stop("Unknown 'stitch_direction'.")
  }

  # Load the images. -----------------------------------------------------------
  # If only paths to images are provided.
  if(is.null(imgs)){
    imgs <- lapply(img_paths, png::readPNG)
  } else { # If images are already loaded.
    # If the images are only matrices, give them three layers.
    imgs <- lapply(imgs, function(curr_img){
      if(length(dim(curr_img))==2){
        curr_img <- array(c(curr_img,curr_img,curr_img),
                          dim = c(dim(curr_img),3))
      }
      return(curr_img)
    })
  }

  # Rotate the images according to the stitch direction, to always have
  # left to right. -------------------------------------------------------------
  if(stitch_direction == "bottom_to_top"){
    imgs <- lapply(1:length(imgs),
                   function(X){
                     return(rotate_clockwise(imgs[[X]], degrees = 90))
                   })
  } else if(stitch_direction == "top_to_bottom"){
    imgs <- lapply(1:length(imgs),
                   function(X){
                     return(rotate_clockwise(imgs[[X]], degrees = 270))
                   })
  } else if(stitch_direction == "right_to_left"){
    imgs <- lapply(1:length(imgs),
                   function(X){
                     return(rotate_clockwise(imgs[[X]], degrees = 180))
                   })
  }

  # Check input PART II. -------------------------------------------------------
  if(is.null(overlap_px) ||
     sum(overlap_px<0 | overlap_px > sapply(imgs, dim)[2,])>0 ||
     length(overlap_px) != (length(imgs) - 1)) {
    stop("No (usable) 'overlap_px' specified.")
  }
  if(length(imgs)<2){
    stop("At least two images needed.")
  }

  # Start stitching. -----------------------------------------------------------
  stitched_img <- imgs[[1]]

  for(i in 2:length(imgs)){
    next_img <- imgs[[i]]
    ovrlp_width <- overlap_px[i - 1]  # Get the overlap width for this pair.
    # Define the overlap region (stitch left to right).
    curr_img_width <- dim(stitched_img)[2]
    if(ovrlp_width>0){
      # Columns of overlap in current stitched and next image:
      ovrlp_stitched <- (curr_img_width - ovrlp_width + 1):curr_img_width
      ovrlp_next_img <- 1:ovrlp_width
      # Apply blending to the overlap region.
      if (blending_mode == "under") {
        # Do nothing.
      } else if (blending_mode == "over") {
        stitched_img[,ovrlp_stitched,] <- next_img[,ovrlp_next_img,]
      } else if (blending_mode == "average") {
        stitched_img[,ovrlp_stitched, ] <-
          (stitched_img[,ovrlp_stitched,] + next_img[,ovrlp_next_img,]) / 2
      } else if (blending_mode == "max") {
        stitched_img[,ovrlp_stitched,] <-
          pmax(stitched_img[,ovrlp_stitched,], next_img[,ovrlp_next_img,])
      } else if (blending_mode == "min") {
        stitched_img[,ovrlp_stitched,] <-
          pmin(stitched_img[,ovrlp_stitched,], next_img[,ovrlp_next_img,])
      }
    }
    right_side_next_img <- (ovrlp_width + 1):dim(next_img)[2]
    # Attach the non-overlapping part of the next image.
    non_overlap_next <- next_img[,right_side_next_img,]
    stitched_img <- abind::abind(stitched_img, non_overlap_next, along = 2)
  }
  # Rotate the image back to the original stitch direction. --------------------
  if(stitch_direction == "bottom_to_top"){
    stitched_img <- rotate_clockwise(stitched_img, degrees = 270)
  } else if(stitch_direction == "top_to_bottom"){
    stitched_img <- rotate_clockwise(stitched_img, degrees = 90)
  } else if(stitch_direction == "right_to_left"){
    stitched_img <- rotate_clockwise(stitched_img, degrees = 180)
  }

  # Add the overlap values and save the image if specified. --------------------
  if(!is.null(out_file)) {
    png::writePNG(stitched_img, paste0(out_file))
  }
  attr(stitched_img, "overlap") <- overlap_px
  #attributes(stitched_img)
  return(stitched_img)
}
# Auxiliary function
rotate_clockwise <- function(img_array, degrees = c(90, 180, 270)) {
  if(!degrees %in% c(90, 180, 270)){
    stop("Value for 'degrees' no allowed.")
  }
  # Initialize new array:
  if(degrees == 90 || degrees == 270){
    rotated_array <- array(0, dim = c(dim(img_array)[2],
                                      dim(img_array)[1],
                                      dim(img_array)[3]))
  } else { # degrees == 180
    rotated_array <- array(0, dim = dim(img_array))
  }

  # Apply rotation to each channel separately.
  for (i in 1:dim(img_array)[3]) {
    if(degrees == 90){
      rotated_array[,,i] <- t(apply(img_array[,,i], 2, rev))
    } else if(degrees == 180){
      rotated_array[,,i] <- t(apply(t(apply(img_array[,,i], 2, rev)), 2, rev))
    } else { # degrees == 270
      rotated_array[,,i] <- apply(t(img_array[,,i]), 2, rev)
    }
  }
  return(rotated_array)
}
