% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/as_caugi.R
\name{as_caugi}
\alias{as_caugi}
\title{Convert to a \code{caugi}}
\usage{
as_caugi(
  x,
  class = c("DAG", "PDAG", "PAG", "UNKNOWN"),
  simple = TRUE,
  build = TRUE,
  collapse = FALSE,
  collapse_to = "---",
  ...
)
}
\arguments{
\item{x}{An object to convert to a \code{caugi}.}

\item{class}{"DAG", "PDAG", "PAG", or "UNKNOWN".
"PAG" is only supported for integer coded matrices.}

\item{simple}{logical. If \code{TRUE} (default) the graph will be simple
(no multiple edges or self-loops).}

\item{build}{logical. If \code{TRUE} (default) build the graph now, otherwise
build lazily on first query or when using \code{\link[=build]{build()}}.}

\item{collapse}{logical. If \code{TRUE} collapse mutual directed edges to
undirected edges. Default is \code{FALSE}.}

\item{collapse_to}{Character string to use as the edge glyph when collapsing.
Should be a registered symmetrical edge glyph. Default is \code{"---"}.}

\item{...}{Additional arguments passed to specific methods.}
}
\value{
A \code{caugi} object.
}
\description{
Convert an object to a \code{caugi}. The object can be a
\code{graphNEL}, \code{matrix}, \code{tidygraph}, \code{daggity}, \code{bn}, or \code{igraph}.
}
\details{
For matrices, \code{as_caugi} assumes that the rows are the \code{from} nodes
and the columns are the \code{to} nodes. Thus, for a graph, G: A --> B, we would
have that \code{G["A", "B"] == 1} and \code{G["B", "A"] == 0}.
For PAGs, the integer codes are as follows (as used in \code{pcalg}):
\itemize{
\item 0: no edge
\item 1: circle (e.g., \verb{A o-o B} or \verb{A o-- B})
\item 2: arrowhead (e.g., \verb{A --> B} or \verb{A o-> B})
\item 3: tail (e.g., \verb{A o-- B} or \code{A --- B})
}
}
\examples{
# igraph
ig <- igraph::graph_from_literal(A - +B, B - +C)
cg_ig <- as_caugi(ig, class = "DAG")

# graphNEL
gn <- graph::graphNEL(nodes = c("A", "B", "C"), edgemode = "directed")
gn <- graph::addEdge("A", "B", gn)
gn <- graph::addEdge("B", "C", gn)
cg_gn <- as_caugi(gn, class = "DAG")

# adjacency matrix
m <- matrix(0L, 3, 3, dimnames = list(LETTERS[1:3], LETTERS[1:3]))
m["A", "B"] <- 1L
m["B", "C"] <- 1L
cg_adj <- as_caugi(m, class = "DAG")

# bnlearn
bn <- bnlearn::model2network("[A][B|A][C|B]")
cg_bn <- as_caugi(bn, class = "DAG")

# dagitty
dg <- dagitty::dagitty("dag {
 A -> B
 B -> C
 }")
cg_dg <- as_caugi(dg, class = "DAG")

cg <- caugi(A \%-->\% B \%-->\% C, class = "DAG")

# check that all nodes are equal in all graph objects
for (cg_converted in list(cg_ig, cg_gn, cg_adj, cg_bn, cg_dg)) {
  stopifnot(identical(nodes(cg), nodes(cg_converted)))
  stopifnot(identical(edges(cg), edges(cg_converted)))
}

# collapse mutual edges
ig2 <- igraph::graph_from_literal(A - +B, B - +A, C - +D)
cg2 <- as_caugi(ig2, class = "PDAG", collapse = TRUE, collapse_to = "---")

# coded integer matrix for PAGs (pcalg style)
nm <- c("A", "B", "C", "D")
M <- matrix(0L, 4, 4, dimnames = list(nm, nm))

# A --> B
M["A", "B"] <- 2L # mark at B end
M["B", "A"] <- 3L # mark at A end

# A --- C
M["A", "C"] <- 3L
M["C", "A"] <- 3L

# B o-> C
M["B", "C"] <- 2L
M["C", "B"] <- 1L

# C o-o D
M["C", "D"] <- 1L
M["D", "C"] <- 1L

cg <- as_caugi(M, class = "PAG")

}
\seealso{
Other conversions: 
\code{\link{as_adjacency}()},
\code{\link{as_bnlearn}()},
\code{\link{as_dagitty}()},
\code{\link{as_igraph}()}
}
\concept{conversions}
