#' @title
#' covselec
#'
#' @description This function is used for state-wise weighted biomarker selection for both lasso and fused type penalty
#' and is applicable for a multi-state model encompassing finite number of states and transitions.
#'
#'
#' @details
#' This function deals with biomarker selection for each state of a multi-state model each
#' phase involves selecting significant biomarkers for that state from two sets: one comprising covariates
#' from the preceding state, and another consisting of all covariates from the current state that were
#' previously unselected.If the total number of biomarkers is referred to as the overall count, and each
#' state in a multi-state model has its own number of selected biomarkers, then in a model with four states,
#' the number of selected biomarkers for the fourth state is determined using a weighted combination.
#' Specifically, it depends on the number of biomarkers selected in the third state, a modified set from the
#' second state, another modified set from the first state
#'
#'
#' @param data Multi state dataset with multiple state and time column in the data
#' @param time_cols time-to-event columns for each state in a MSM
#' @param status_cols status columns for corresponding states in a MSM
#' @param covariate_range range of high-dimensional biomarkers in the MSM dataset
#' @param alphas penalty parameter for lasso type penalty
#' @param lambdas penalty parameter for fused type penalty
#' @param method method to be selected for covariate selction either "lasso" or "fused"
#' @param p_cut association between variables
#' @param verbose Logical indicating whether to print progress messages
#' @return list containing selected biomarkers for each state
#' @import  glmnet
#'          mstate
#'
#' @importFrom stats coef coefficients
#' @importFrom penalized penalized
#' @importFrom survival Surv coxph survfit
#'
#' @examples
#' ##
#' set.seed(123)
#' covselec(data= highDmsm,time_cols = c("state1", "state2","state3"),
#' status_cols = c("status1", "status2","status3"),
#' covariate_range = 8:107,alphas=c(0.40,0.45,0.60),
#' lambdas = c(0.1,0.15,0.20),method = "lasso", p_cut = 0.2,verbose = TRUE)
#' ##
#' @export
#' @author Atanu Bhattacharjee,Gajendra Kumar Vishwakarma,Abhipsa Tripathy


covselec <- function(data, time_cols, status_cols, covariate_range,
                     alphas, lambdas, method = "lasso", p_cut = 0.2,verbose = TRUE) {

  if (length(alphas) != length(time_cols)) {
    stop("The number of alpha values must be equal to the number of states.")
  }
  if (length(lambdas) != length(time_cols)) {
    stop("The number of lambda values must be equal to the number of states.")
  }

  selected_predictors <- vector("list", length(time_cols))

  for (state in seq_along(time_cols)) {
    if (verbose) {
      message("Processing state: ", state)
    }
    
    # Step 1 — Univariate Cox screening
    pvals <- sapply(covariate_range, function(j) {
      fit <- coxph(Surv(data[[time_cols[state]]], data[[status_cols[state]]]) ~ data[[j]])
      summary(fit)$coefficients[,"Pr(>|z|)"]
    })
    keep_vars <- covariate_range[pvals < p_cut]

    if (length(keep_vars) == 0) {
      selected_predictors[[state]] <- character(0)
      next
    }

    # Prepare design matrix
    X <- scale(as.matrix(data[, keep_vars, drop = FALSE]))
    Y <- Surv(data[[time_cols[state]]], data[[status_cols[state]]])

    # Step 2 — Method choice
    if (method == "lasso") {
      # Fit penalized Cox via glmnet
      fit <- glmnet(x = X, y = Y, family = "cox", alpha = alphas[state])
      cv_fit <- cv.glmnet(x = X, y = Y, family = "cox", alpha = alphas[state])
      coefs <- coef(cv_fit, s = "lambda.min")

      # If too few selected, relax lambda
      if (sum(coefs != 0) <= 1) {
        lambda_seq <- exp(seq(log(min(cv_fit$lambda)), log(min(cv_fit$lambda) / 50), length = 50))
        fit2 <- glmnet(x = X, y = Y, family = "cox", alpha = alphas[state], lambda = lambda_seq)
        coefs <- coef(fit2, s = min(lambda_seq))
      }

    } else if (method == "fusionlasso") {
      # Fit penalized Cox via penalized package with fused lasso
      fit <- penalized(response = Y, penalized = X,
                       lambda1 = alphas[state], lambda2 = lambdas[state],
                       model = "cox", fusedl = TRUE)
      coefs <- coefficients(fit)
    } else {
      stop("Unsupported method. Choose either 'lasso' or 'fusionlasso'.")
    }

    # Step 3 — Extract selected covariates
    if (inherits(coefs, "dgCMatrix")) {
      sel_covs <- rownames(coefs)[as.numeric(coefs) != 0]
    } else {
      sel_covs <- names(coefs)[coefs != 0]
    }

    sel_covs <- sel_covs[sel_covs != "(Intercept)"]
    selected_predictors[[state]] <- sel_covs
  }

  return(selected_predictors)
}










