% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/write_testthat_file.R
\name{write_testthat_file}
\alias{write_testthat_file}
\title{Transform a \strong{pointblank} agent to a \strong{testthat} test file}
\usage{
write_testthat_file(
  agent,
  name = NULL,
  path = NULL,
  overwrite = FALSE,
  skips = NULL,
  quiet = FALSE
)
}
\arguments{
\item{agent}{\emph{The pointblank agent object}

\verb{obj:<ptblank_agent>} // \strong{required}

A \strong{pointblank} \emph{agent} object that is commonly created through the use of
the \code{\link[=create_agent]{create_agent()}} function.}

\item{name}{\emph{Name for generated testthat file}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional name for for the \strong{testhat} test file. This should be a name
without extension and without the leading \code{"test-"} text. If nothing is
supplied, the name will be derived from the \code{tbl_name} in the agent. If
that's not present, a generic name will be used.}

\item{path}{\emph{File path}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

A path can be specified here if there shouldn't be an attempt to place the
file in \code{testthat/tests}.}

\item{overwrite}{\emph{Overwrite a previous file of the same name}

\verb{scalar<logical>} // \emph{default:} \code{FALSE}

Should a \strong{testthat} file of the same name be overwritten?}

\item{skips}{\emph{Test skipping}

\verb{vector<character>} // \emph{default:} \code{NULL} (\code{optional})

This is an optional vector of test-skipping keywords modeled after the
\strong{testthat} \verb{skip_on_*()} functions. The following keywords can be used to
include \verb{skip_on_*()} statements: \code{"cran"} (\code{\link[testthat:skip]{testthat::skip_on_cran()}}),
\code{"travis"} (\code{\link[testthat:skip_on_travis]{testthat::skip_on_travis()}}), \code{"appveyor"}
(\code{\link[testthat:skip_on_travis]{testthat::skip_on_appveyor()}}), \code{"ci"} (\code{\link[testthat:skip]{testthat::skip_on_ci()}}),
\code{"covr"} (\code{\link[testthat:skip]{testthat::skip_on_covr()}}), \code{"bioc"}
(\code{\link[testthat:skip]{testthat::skip_on_bioc()}}). There are keywords for skipping tests on
certain operating systems and all of them will insert a specific
\code{\link[testthat:skip]{testthat::skip_on_os()}} call. These are \code{"windows"}
(\code{skip_on_os("windows")}), \code{"mac"} (\code{skip_on_os("mac")}), \code{"linux"}
(\code{skip_on_os("linux")}), and \code{"solaris"} (\code{skip_on_os("solaris")}). These
calls will be placed at the top of the generated \strong{testthat} test file.}

\item{quiet}{\emph{Inform (or not) upon file writing}

\verb{scalar<logical>} // \emph{default:} \code{FALSE}

Should the function \emph{not} inform when the file is written?}
}
\value{
Invisibly returns \code{TRUE} if the \strong{testthat} file has been written.
}
\description{
With a \strong{pointblank} \emph{agent}, we can write a \strong{testthat} test file and opt
to place it in the \code{testthat/tests} if it is available in the project path
(we can specify an alternate path as well). This works by transforming the
validation steps to a series of \verb{expect_*()} calls inside individual
\code{\link[testthat:test_that]{testthat::test_that()}} statements.

A major requirement for using \code{write_testthat_file()} on an agent is the
presence of an expression that can retrieve the target table. Typically, we
might supply a table-prep formula, which is a formula that can be invoked to
obtain the target table (e.g., \code{tbl = ~ pointblank::small_table}). This
user-supplied statement will be used by \code{write_testthat_file()} to generate a
table-loading statement at the top of the new \strong{testthat} test file so that
the target table is available for each of the \code{\link[testthat:test_that]{testthat::test_that()}}
statements that follow. If an \emph{agent} was not created using a table-prep
formula set for the \code{tbl}, it can be modified via the \code{\link[=set_tbl]{set_tbl()}} function.

Thresholds will be obtained from those applied for the \code{stop} state. This can
be set up for a \strong{pointblank} \emph{agent} by passing an \code{action_levels} object
to the \code{actions} argument of \code{\link[=create_agent]{create_agent()}} or the same argument of any
included validation function. If \code{stop} thresholds are not available, then a
threshold value of \code{1} will be used for each generated \verb{expect_*()} statement
in the resulting \strong{testthat} test file.

There is no requirement that the \strong{agent} first undergo interrogation with
\code{\link[=interrogate]{interrogate()}}. However, it may be useful as a dry run to interactively
perform an interrogation on the target data before generating the
\strong{testthat} test file.
}
\details{
Tests for inactive validation steps will be skipped with a clear message
indicating that the reason for skipping was due to the test not being active.
Any inactive validation steps can be forced into an active state by using the
\code{\link[=activate_steps]{activate_steps()}} on an \emph{agent} (the opposite is possible with the
\code{\link[=deactivate_steps]{deactivate_steps()}} function).

The \strong{testthat} package comes with a series of \verb{skip_on_*()} functions which
conveniently cause the test file to be skipped entirely if certain conditions
are met. We can quickly set any number of these at the top of the
\strong{testthat} test file by supplying keywords as a vector to the \code{skips}
option of \code{write_testthat_file()}. For instance, setting
\verb{skips = c("cran", "windows)} will add the \strong{testthat} \code{skip_on_cran()} and
\code{skip_on_os("windows")} statements, meaning that the generated test file
won't run on a CRAN system or if the system OS is Windows.

Here is an example of \strong{testthat} test file output (\code{"test-small_table.R"}):

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Generated by pointblank

tbl <- small_table

test_that("column `date_time` exists", \{

  expect_col_exists(
    tbl,
    columns = date_time,
    threshold = 1
  )
\})

test_that("values in `c` should be <= `5`", \{

  expect_col_vals_lte(
    tbl,
    columns = c,
    value = 5,
    threshold = 0.25
  )
\})

}\if{html}{\out{</div>}}

This was generated by the following set of R statements:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(pointblank)

agent <-
  create_agent(
    tbl = ~ small_table,
    actions = action_levels(stop_at = 0.25)
  ) \%>\%
  col_exists(date_time) \%>\%
  col_vals_lte(c, value = 5)

write_testthat_file(
  agent = agent,
  name = "small_table",
  path = "."
)
}\if{html}{\out{</div>}}
}
\section{Examples}{

\subsection{Creating a \strong{testthat} file from an \emph{agent}}{

Let's walk through a data quality analysis of an extremely small table. It's
actually called \code{small_table} and we can find it as a dataset in this
package.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table
#> # A tibble: 13 x 8
#>    date_time           date           a b             c      d e     f    
#>    <dttm>              <date>     <int> <chr>     <dbl>  <dbl> <lgl> <chr>
#>  1 2016-01-04 11:00:00 2016-01-04     2 1-bcd-345     3  3423. TRUE  high 
#>  2 2016-01-04 00:32:00 2016-01-04     3 5-egh-163     8 10000. TRUE  low  
#>  3 2016-01-05 13:32:00 2016-01-05     6 8-kdg-938     3  2343. TRUE  high 
#>  4 2016-01-06 17:23:00 2016-01-06     2 5-jdo-903    NA  3892. FALSE mid  
#>  5 2016-01-09 12:36:00 2016-01-09     8 3-ldm-038     7   284. TRUE  low  
#>  6 2016-01-11 06:15:00 2016-01-11     4 2-dhe-923     4  3291. TRUE  mid  
#>  7 2016-01-15 18:46:00 2016-01-15     7 1-knw-093     3   843. TRUE  high 
#>  8 2016-01-17 11:27:00 2016-01-17     4 5-boe-639     2  1036. FALSE low  
#>  9 2016-01-20 04:30:00 2016-01-20     3 5-bce-642     9   838. FALSE high 
#> 10 2016-01-20 04:30:00 2016-01-20     3 5-bce-642     9   838. FALSE high 
#> 11 2016-01-26 20:07:00 2016-01-26     4 2-dmx-010     7   834. TRUE  low  
#> 12 2016-01-28 02:51:00 2016-01-28     2 7-dmx-010     8   108. FALSE low  
#> 13 2016-01-30 11:23:00 2016-01-30     1 3-dka-303    NA  2230. TRUE  high
}\if{html}{\out{</div>}}

Creating an \code{action_levels} object is a common workflow step when creating a
pointblank agent. We designate failure thresholds to the \code{warn}, \code{stop}, and
\code{notify} states using \code{action_levels()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{al <-
  action_levels(
    warn_at = 0.10,
    stop_at = 0.25,
    notify_at = 0.35
  )
}\if{html}{\out{</div>}}

A pointblank \code{agent} object is now created and the \code{al} object is provided to
the agent. The static thresholds provided by the \code{al} object make reports a
bit more useful after interrogation.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent <-
  create_agent(
    tbl = ~ small_table,
    label = "An example.",
    actions = al
  ) \%>\%
  col_exists(c(date, date_time)) \%>\%
  col_vals_regex(
    b,
    regex = "[0-9]-[a-z]\{3\}-[0-9]\{3\}"
  ) \%>\%
  col_vals_gt(d, value = 100) \%>\%
  col_vals_lte(c, value = 5) \%>\%
  interrogate()
}\if{html}{\out{</div>}}

This agent and all of the checks can be transformed into a testthat file with
\code{write_testthat_file()}. The \code{stop} thresholds will be ported over to the
\verb{expect_*()} functions in the new file.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{write_testthat_file(
  agent = agent,
  name = "small_table",
  path = "."
)
}\if{html}{\out{</div>}}

The above code will generate a file with the name \code{"test-small_table.R"}. The
path was specified with \code{"."} so the file will be placed in the working
directory. If you'd like to easily add this new file to the \code{tests/testthat}
directory then \code{path = NULL} (the default) will makes this possible (this is
useful during package development).

What's in the new file? This:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Generated by pointblank

tbl <- small_table

test_that("column `date` exists", \{

  expect_col_exists(
    tbl,
    columns = date,
    threshold = 1
  )
\})

test_that("column `date_time` exists", \{

  expect_col_exists(
    tbl,
    columns = date_time,
    threshold = 1
  )
\})

test_that("values in `b` should match the regular expression:
`[0-9]-[a-z]\{3\}-[0-9]\{3\}`", \{

  expect_col_vals_regex(
    tbl,
    columns = b,
    regex = "[0-9]-[a-z]\{3\}-[0-9]\{3\}",
    threshold = 0.25
  )
\})

test_that("values in `d` should be > `100`", \{

  expect_col_vals_gt(
    tbl,
    columns = d,
    value = 100,
    threshold = 0.25
  )
\})

test_that("values in `c` should be <= `5`", \{

  expect_col_vals_lte(
    tbl,
    columns = c,
    value = 5,
    threshold = 0.25
  )
\})
}\if{html}{\out{</div>}}
}

\subsection{Using an \emph{agent} stored on disk as a YAML file}{

An agent on disk as a YAML file can be made into a \strong{testthat} file. The
\code{"agent-small_table.yml"} file is available in the \strong{pointblank} package
and the path can be obtained with \code{system.file()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{yml_file <-
  system.file(
    "yaml", "agent-small_table.yml",
    package = "pointblank"
  )
}\if{html}{\out{</div>}}

Writing the \strong{testthat} file into the working directory is much the same as
before but we're reading the agent from disk this time. It's a read and write
combo, really. Again, we are choosing to write the file to the working
directory by using \code{path = "."}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{write_testthat_file(
  agent = yaml_read_agent(yml_file),
  name = "from_agent_yaml",
  path = "."
)
}\if{html}{\out{</div>}}
}
}

\section{Function ID}{

8-5
}

\seealso{
Other Post-interrogation: 
\code{\link{all_passed}()},
\code{\link{get_agent_x_list}()},
\code{\link{get_data_extracts}()},
\code{\link{get_sundered_data}()}
}
\concept{Post-interrogation}
