#' @title Load 'FreeSurfer' brain from 'RAVE'
#' @description Create 3D visualization of the brain and visualize with
#' modern web browsers
#' @param subject character, list, or \code{\link{RAVESubject}} instance; for
#' list or other objects, make sure \code{subject$subject_id} is a valid 'RAVE'
#' subject 'ID'
#' @param surfaces one or more brain surface types from \code{"pial"},
#' \code{"white"}, \code{"smoothwm"}, \code{"pial-outer-smoothed"}, etc.;
#' check \code{\link[threeBrain]{threeBrain}}
#' @param overlays volumes to overlay; default is \code{'aparc.a2009s+aseg'}
#' @param annotations surface annotation or curvature data to load;
#' default is \code{'label/aparc.a2009s'}, referring to the
#' \code{'*h.aparc.a2009s.annot'} under the label folder.
#' @param usetemplateifmissing whether to use template brain when the subject
#' brain files are missing. If set to true, then a template (usually 'N27')
#' brain will be displayed as an alternative solution, and electrodes will be
#' rendered according to their \code{'MNI305'} coordinates, or
#' \code{'VertexNumber'} if given.
#' @param include_electrodes whether to include electrode in the model; default
#' is true
#' @param ... ignored, reserved for legacy code
#' @returns A \code{'threeBrain'} instance if brain is found or
#' \code{usetemplateifmissing} is set to true; otherwise returns \code{NULL}
#' @examples
#'
#'
#' if(has_rave_subject("demo/DemoSubject")) {
#'
#'   brain <- rave_brain("demo/DemoSubject")
#'
#'   if(interactive()) {
#'
#'     brain$plot()
#'
#'   }
#'
#' }
#'
#'
#' @export
rave_brain <- function(
    subject, surfaces = 'pial', overlays = "aparc.a2009s+aseg", annotations = "label/aparc.a2009s",
    ..., usetemplateifmissing = FALSE, include_electrodes = TRUE){

  subject <- as_rave_subject(subject, strict = FALSE)

  fs_path <- subject$freesurfer_path

  electrode_table <- NULL
  if(include_electrodes) {
    electrode_table <- subject$meta_data(meta_type = "electrodes", strict = FALSE)
    if(!is.data.frame(electrode_table) || !nrow(electrode_table)) {
      electrode_table <- NULL
    }
  }


  if(is.na(fs_path) || !isTRUE(dir_exists(fs_path))){
    if( !usetemplateifmissing ){
      return(invisible())
    }
    brain <- threeBrain::merge_brain(
      template_surface_types = surfaces,
      template_atlas_types = overlays,
      template_annotation_types = annotations
    )

    if(is.data.frame(electrode_table)) {
      # try to use MNI305 position
      if(all(paste0("MNI305_", c("x", "y", "z")) %in% names(electrode_table))){
        electrode_table$Coord_x <- electrode_table$MNI305_x
        electrode_table$Coord_y <- electrode_table$MNI305_y
        electrode_table$Coord_z <- electrode_table$MNI305_z
      }

      brain$set_electrodes(electrodes = electrode_table)
    }

  } else {
    # if(recache){
    #   if( clean_before_cache ){
    #     fs <- list.files(file.path(fs_path, 'RAVE'), pattern = '\\.json$',
    #                     all.files = FALSE, recursive = FALSE, full.names = TRUE,
    #                     ignore.case = TRUE, include.dirs = FALSE, no.. = TRUE)
    #     lapply(fs, unlink)
    #   }
    #   threeBrain::import_from_freesurfer(fs_path, subject_name = subject$subject_code)
    # }

    brain <- threeBrain::threeBrain(
      path = fs_path,
      subject_code = subject$subject_code,
      surface_types = surfaces,
      atlas_types = overlays,
      annotation_types = annotations
    )


    if(is.data.frame(electrode_table)) {
      brain$set_electrodes(electrodes = electrode_table)
    }

    # if( compute_template ){
    #   tf <- tempfile()
    #   new_table <- brain$calculate_template_coordinates(save_to = tf)
    #   if( file.exists(tf) ){
    #     brain$electrodes$raw_table_path <- NULL
    #     unlink(tf)
    #     # need to update meta
    #     save_meta2(
    #       data = new_table,
    #       meta_type = 'electrodes',
    #       project_name = subject$project_name,
    #       subject_code = subject$subject_code
    #     )
    #   }
    # }

  }

  brain$meta$constructor_params <- list(
    project_name = subject$project_name,
    subject_code = subject$subject_code,
    use_141 = FALSE,
    usetemplateifmissing = usetemplateifmissing
  )

  brain
}
