
#'Get the labels from a dataset
#'
#' This function get the labels from a dataset
#'
#' @param data Dataset
#' @param vars Variables of interest
#' @return The labels
#' @keywords internal

get_labels <- function(data, vars) {
  labels <- lapply(vars, function(var) {
    # Ensure column exists
    if (!var %in% names(data)) return(var)

    # Use tryCatch in case attr access throws errors on weird types
    lbl <- tryCatch({
      attr(data[[var]], "label")
    }, error = function(e) NULL)

    # If still NULL, attempt labelled::var_label if available
    if (is.null(lbl) && requireNamespace("labelled", quietly = TRUE)) {
      lbl <- labelled::var_label(data[[var]])
      if (is.list(lbl)) lbl <- unlist(lbl)  # var_label returns a named list
    }

    # Final check
    if (!is.null(lbl) && is.character(lbl) && nzchar(lbl)) {
      return(lbl)
    } else {
      return(var)
    }
  })

  names(labels) <- vars
  return(labels)
}



#'Geometric mean
#'
#' This function return the geometric mean
#'
#' @param x Numeric vector
#' @return Geometric mean
#' @keywords internal

geom_mean <- function(x, na.rm = TRUE) {
  exp(mean(log(x), na.rm = na.rm))
}



#' Standard error
#'
#' This function return the standard error
#'
#' @param x Numeric vector
#' @return standard error
#' @keywords internal
se <- function(x) stats::sd(x)/sqrt(length(x))




format_lookup <- list(
  mean_sd = "{mean} ({sd})",
  mean_se = "{mean} ({se})",
  median_range = "{median} ({min}, {max})",
  median_IQR = "{median} ({p25}, {p75})",
  geomMean_sd = "{geom_mean} ({sd})"
)

format_lookup_cat <-
  list(
    n_percent = "{n} ({p}%)",
    n = "{n}",
    n_N = "{n}/{N}"
  )


add_by_n <- function(data, variable, by, ...) {
  data |>
    dplyr::select(all_of(c(variable, by))) |>
    dplyr::arrange(pick(all_of(c(by, variable)))) |>
    dplyr::group_by(.data[[by]]) |>
    dplyr::summarise_all(~sum(!is.na(.))) |>
    rlang::set_names(c("by", "variable")) |>
    dplyr::mutate(
      by_col = paste0("add_n_stat_", dplyr::row_number()),
      variable = style_number(variable)
    ) %>%
    select(-by) %>%
    tidyr::pivot_wider(names_from = by_col,
                       values_from = variable)
}

FitFlextableToPage <- function(ft, pgwidth = 6){
  ft_out <- ft %>% flextable::autofit()
  ft_out <- flextable::width(ft_out, width = dim(ft_out)$widths*pgwidth /(flextable::flextable_dim(ft_out)$widths))
  return(ft_out)
}

